// ********************************* Tilia.h **************************************

//
// Tilia Music Representation
//
// This was written by Lippold Haken and Dorothea Blostein.
//
// Copyright (c) Lippold Haken and Dorothea Blostein, 1993 - 2013.
//
// We encourage distribution of this C header file along with the
// Lime application and the Lime User's Manual.
//
// Incorporation of this C header file into other programs is encouraged.  The Lime
// authors do not retain any rights to such software.
//
// For an introduction to the Tilia Music Representation, see "The Tilia Music
// Representation: Extensibility, Abstraction, and Notation Contexts for the Lime
// Music Editor," Computer Music Journal, Vol 17, No 3, Fall 1993, pp.43-58.
//
#ifndef TILIAH__ADCF4890_3CBB_11E2_A25F_2B0AF6D25AD4__INCLUDED_
#define TILIAH__ADCF4890_3CBB_11E2_A25F_2B0AF6D25AD4__INCLUDED_


//
// This file contains the type definitions for the Tilia music representation.
// The Tilia music representation is used by the Lime music notation program.
// These type definitions contain the information you need to be able to read,
// modify or create Tilia files on disk.
//
// A Tilia file is written in "binary" (non-ascii) format as follows:
//		header
//		Sequence of Tilia nodes for first voice
//		Sequence of Tilia nodes for second voice
//		...
//		Sequence of Tilia nodes for last voice
// The header defines the number of voices in the piece, as well as other information.
// A complete description of the header contents, as well as Tilia node format,
// is given later in this file.
//
// As an overview, the format of the file header is
//		file-format number  (2 bytes)
//		H					(2 bytes)
//		other info			(H bytes)
// H is variable here; when H is small, the unspecified quantities are given default
// values by Lime when the piece is opened.  All pieces created by Lime are given
// full headers (H is large).  In the future, the header may be extendeded to contain
// more information; thus H will increase.
//
// The minimal header is six bytes long and consists of
//		file-format number  (2 bytes)		set this to CURRENT_FILE_FORMAT_NUMBER
//		2					(2 bytes)
//		no. voices			(2 bytes)
//

// All Tilia structures depend on 2-byte (16-bit) packing alignment in structures.
// This may need to be set as a compiler option on some compilers.
// For Windows Visual C and for GNU & LLVM compilers (used on Mac)
// a #pragma, such as the following is needed (restored at the end of this file):
//
#if defined( __BORLANDC__ )
	#pragma option -a2
#else
	#pragma pack(push, 2)
#endif



// ---------------------- CURRENT_FILE_FORMAT_NUMBER ----------------------------
//
// As a general principle, a version of Lime should always be able to read the
// next version of a Tilia file, albeit with the possibility of information loss.
// If a change to the format is made that will mean that the previous version of
// Lime cannot possibly read it, then the file-format-number should be increased
// by 2 (two) and, if practical, an interim version produced that can read it.
//
#define CURRENT_FILE_FORMAT_NUMBER	345

#define FIRST_56_BYTE_NODE_VERSION	343
#define GENERAL_RELATIVITY_VERSION	345	// Format number supporting general relativity
#define UNICODE_ANNOTATION_VERSION	346	// Format number defaulting to UTF-8 annotations
#define FIRST_52_BYTE_NODE_VERSION	347

#define NEWEST_READABLE_INCREMENT	1

//
// ---------------------- Generic type definitions ------------------------------
// Implementations of the C language vary in the number of bits that are allocated
// for types such as "int" and "long".  In order to avoid confusion, we declare
// macros for signed and unsigned integers of 8, 16 and 32 bits, as well as for
// floating point numbers of 32 bits.  64 bit integers are not used in any node
// on disk, but if it can be defined, we do so.
//
// Two sets of definitions are supplied here: for compiling Lime to run under
// Windows and for compiling Lime to run on a real Macintosh.
//

#if COMPILE_FOR_WINDOWS			// Definitions for imitation Mac under 32-bit Windows
	typedef signed		char	INT8;		// INT8	   -- signed	8-bit integer
	typedef unsigned	char	UNINT8;		// UNINT8  -- unsigned  8-bit integer
	typedef				short	INT16;		// INT16   -- signed   16-bit integer
	typedef unsigned	short	UNINT16;	// UNINT16 -- unsigned 16-bit integer
	typedef				int		INT32;		// INT32   -- signed   32-bit integer
	typedef unsigned	long	UNINT32;	// UNINT32 -- unsigned 32-bit integer
# ifndef INT64
	typedef		long	long	INT64;		// INT64   -- signed   64-bit integer
	typedef unsigned  long long	UNINT64;	// UNINT64 -- unsigned 64-bit integer
# endif
	typedef unsigned	char	PASCH;		// PASCH   -- pascal character
	typedef unsigned	char*	PSTRING;	// PSTRING -- pascal string

	typedef				INT32	Coord;		// Coord   -- screen coordinate
	typedef				float	REAL32;		// REAL32  -- 32-bit floating point number
	typedef				double	REAL64;		// REAL64  -- 64-bit floating point number

#else // COMPILE_FOR_INTELMAC || COMPILE_FOR_POWERMAC
# if __i386__
#  define COMPILE_FOR_POWERMAC	0
#  define COMPILE_FOR_INTELMAC	1
# else
#  define COMPILE_FOR_POWERMAC	1
#  define COMPILE_FOR_INTELMAC	0
# endif

	typedef	signed		char	INT8;		// INT8	   -- signed	8-bit integer
	typedef unsigned	char	UNINT8;		// UNINT8  -- unsigned  8-bit integer
	typedef				short	INT16;		// INT16   -- signed   16-bit integer
	typedef unsigned	short	UNINT16;	// UNINT16 -- unsigned 16-bit integer
	typedef				long	INT32;		// INT32   -- signed   32-bit integer
	typedef unsigned	long	UNINT32;	// UNINT32 -- unsigned 32-bit integer
# ifndef INT64
	typedef		long	long	INT64;		// INT64   -- signed   64-bit integer
	typedef unsigned  long long	UNINT64;	// UNINT64 -- unsigned 64-bit integer
# endif
	typedef unsigned	char	PASCH;		// PASCH   -- pascal character
	typedef unsigned	char*	PSTRING;	// PSTRING -- pascal string

	typedef				INT32	Coord;		// Coord   -- screen coordinate (actually on Mac only 16 bits are currently usable)
	typedef				float	REAL32;		// REAL32  -- 32-bit floating point number
	typedef				double	REAL64;		// REAL64  -- 64-bit floating point number
#endif

// On Macs "Style" is defined in <Quickdraw.h> as unsigned char;
// We define it here to make sure that it is, indeed, what we think it is.
// Currently only bold, italic and underline are actually supported.
//
typedef unsigned char		TiliaStyle;
enum {
	tiliaStyle_NORMAL		= 0,
	tiliaStyle_BOLD			= 1,
	tiliaStyle_ITALIC		= 2,
	tiliaStyle_UNDERLINE	= 4,
	tiliaStyle_OUTLINE		= 8,
	tiliaStyle_SHADOW		= 0x10,
	tiliaStyle_CONDENSE		= 0x20,
	tiliaStyle_EXTEND		= 0x40,
	tiliaStyle_KERNED		= 0x80	// reserved
};


// UNBFLD and INBFLD are used for all bit field definitions, unsigned or signed respectively;
// They can be defined to something else if the compiler requires something else.
//
#ifndef UNBFLD
	typedef unsigned char UNBFLD;
#endif
#ifndef INBFLD
	typedef signed   char INBFLD;
#endif


// QQ is used in Lime for foreign language string translation;
// If Tilia.h is being used outside of Lime, it can be dummy.
//
#ifndef  QQ
# define QQ(s)	(s)
#endif



// ------------------------------- Global constants ----------------------------
//
enum{
	MAX08	  = 127,						// biggest positive  8-bit number
	MAX16	  = 32767,						// biggest positive 16 bit number
	MAX32	  = 2147483647L,				// biggest positive 32 bit number
	BIG08	  = 120,						// big  8 bit number that can be exceeded and negated
	BIG16	  = 32000,						// big 16 bit number that can be exceeded and negated
	BIG32	  = 2113929216L,				// big 32 bit number that can be exceeded and negated
	UNDEF08	  = -128,						// undefined  8-bit value
	UNDEF16	  = -32768,						// undefined 16-bit value
	UNDEF32	  = -2147483648LL,				// undefined 32-bit value
	UNDEFXY	  = UNDEF32,					// undefined XYPOS  value
	MAXUN8	  = (UNINT8)0xFF,				// biggest unsigned  8 bit number
	MAXUN16	  = (UNINT16)0xFFFF,			// biggest unsigned 16 bit number
	MAXUN32	  = (UNINT32)0xFFFFFFFF,		// biggest unsigned 32 bit number
	
	BIGXYPOS  = BIG32,						// big   positive XY position value
	MAXXYPOS  = 1048576L					// biggest actual XY position value
};
#undef  INDEFINED
#define INDEFINED(t)	(~(((unsigned t) ((t) -1)) >> 1))	// Undefined value of an arbitrary integer type.

#undef  UNDEFREAL32
#define UNDEFREAL32 (-0.0)
#undef  UNDEFREAL64
#define UNDEFREAL64 (-0.0)

#undef  NIL
#define NIL				0		// legacy - use NULL
#undef  TRUE
#define TRUE			1		// Guaranteed to be 1
#undef	TRUEISH
#define	TRUEISH			(-1)
#undef  FALSE
#define FALSE			0		// Guaranteed to be 0
#undef  FALSEISH
#define FALSEISH		(-128)


// ---------------------------- Sizes ---------------------------
//
enum{
	MAXVC				= 100,		// maximum number of voices in a piece (practical limit is 120)
	MAXPART				= MAXVC,	// maximum number of parts in a piece
	MAXSTF				= MAXVC,	// maximum number of staves on a page (divided among any number of systems).
	MAX_PART_STAVES		= 7,		// Maximum number of staves used by a single part
	
	STF_CHORD_LIMIT		= 200,		// maxmimum number of notes in a chord -- all layers on a staff
	MAX_CHORD			= 10,		// Maximum number of notes in a newly recorded chord
	MAX_ACC_COLUMNS		= 15,		// maximum number of levels of stacked accidentals
#if !COMPILE_FOR_MACOS
	MAXMEAS				= BIG16,	// maximum number of measures for everyone else (absolute max is 32760).
#else
	MAXMEAS				= 4000,		// maximum number of measures for puny OS9 systems
#endif
	MAXNCONTEXTS		= (MAXVC+1),// maximum number of notation contexts: score + each voice

	NOSTAFF				= -1,		// the "staff number" for voices that are not printing now
	NOLIST				= 0,		// the list value if a voice is not listed (deactivated)
	NOPAGE				= UNDEF16	// "page number" for a non-existent page	
};

// Sizes that depend on binary version.
enum{
	TiliaBytesInMemory	= 52,		// number of bytes in Tilia nodes in Memory.
#if CURRENT_FILE_FORMAT_NUMBER >= FIRST_52_BYTE_NODE_VERSION
	TiliaBytes			= TiliaBytesInMemory,	// JA20130104	7.5.12	Prepare for 52 byte Tilia on disk
#else
	TiliaBytes			= TiliaBytesInMemory+4,	// In version 347 and later, 4 fewer bytes will be written to disk
#endif
	OLD_TiliaBytes		= 24,		// before XYPOS_FORMAT
	BytesParamValue		= 14,		// param structure size
	OLD_BytesParamValue	= 10 		// before XYPOS_FORMAT
};
typedef INT32	XYPOS;

typedef struct {
	XYPOS			v;
	XYPOS			h;
} XYPoint;

typedef struct {
	XYPOS			l;
	XYPOS			r;
} XYPair;

typedef struct {
	XYPOS			top;
	XYPOS			left;
	XYPOS			bottom;
	XYPOS			right;
} XYRect;

typedef struct FREAL32 {
	// A FREAL32 is a 32 bit positive real number or a number expressed
	// as the product of 2 fractions.  It occupies 4 bytes.  If the real
	// is positive, it is a straightforward  32 bit positive real number.
	// If it is negative (i.e. the top bit is set), the format is:
	//
	//	 1	3	 4		 12			  12		bits
	//	+----------+-------------------------+
	//	|1|sfn| sfd|  numerator | denominator|
	//	+----------+-------------------------+
	//
	//	denominator	of main fraction			-1	(12 bits)
	//	numerator	of main fraction			-1	(12 bits)
	//	sfd			denominator of sub fraction	-1	(4  bits)
	//	sfn			numerator of sub fraction	-1	(3  bits)
	//	1			1								(1  bit)
	//
	// The type was designed to maintain polytemporal multipliers in a
	// 32 bit field (in TIMESIG nodes).
	//	
	// Unfortunately, with little-ended systems and compilers it is
	// not possible to set up bit fields that put the 12-bit values
	// in the right place, but at the same time keep the decimal
	// values intact (without a special test for the type in endedness
	// conversion); so masking has to be used.  
	//
	union{
		REAL32	d;	// decimal
		INT32	f;	// fractional
	};
} FREAL32;

enum{
	fREAL32_main_mask	= 0xFFF,		// 12 bit mask to extract main values
	fREAL32_main_shift	= 12,			// Right shift to extract numerator
	fREAL32_sub_mask	= 0xF,			// 4 bit mask  to extract sub denominator
	fREAL32_sub_shift	= 24,			// Right shift to extract sub denominator
	fREAL32_ssub_mask	= 0x7,			// 3 bit mask  to extract sub numerator
	fREAL32_ssub_shift	= 28,			// Right shift to extract sub numerator
	fREAL32_zeroffset	= 1				// fraction values are true value -1
};


// Reserved type for flexible handling of colouring
//
typedef struct{
	union{
		UNINT32		rgba;
		
		// The sub structures below are exactly 4 bytes to avoid any compiler alignment issues.
		// The first byte, alpha, is common to both structures in the union.
		//
		struct{
			UNINT8	alpha;				// non-zero means colour
			UNINT8	red, green, blue;	// If alpha non-zero, this is the colour
		};
		
		struct{
			UNINT8	alpha_;				// zero means special (same byte as alpha)
			INT8	opacity;			// Opacity of special colour (0 .. 127);
			//							// If zero then default colour should be used;
			//							// -ve reserved for picture fillers (until then always >=0).
			
			UNINT8	sColor;				// Describe special colour (including 'Paint in White');
			UNINT8	sValue;				// or Picture information  (reserved if opacity -ve).
		};
	};
}TiliaRGBA;
#define TiliaRGBA_sizes "1111"
//
enum{
	ScolorIndexPiece	= 1,			// sValue is index to predefined piece colours
	ScolorIndexLime		= 2,			// sValue is index to predefined Lime  colours 
	ScolorByNoteName	= 3,			// For NOTES or KEYSIGS, special colouring by note name, default otherwise
	ScolorErase			= 128,			// Draw in White - i.e. use background
	ScolorDefault		= 0				// Default colour (i.e. black)
};	



// Identification of Provenance of a Lime Tilia file.
//
enum{
	Tilia_from_UNKNOWN	= 0,
	Tilia_from_POWERMAC	= 1,
	Tilia_from_INTELMAC	= 2,
	Tilia_from_WINDOWS	= 3,
	
	Tilia_from_THIS
#if   COMPILE_FOR_POWERMAC		
	=  Tilia_from_POWERMAC
#elif COMPILE_FOR_INTELMAC
	=  Tilia_from_INTELMAC
#elif COMPILE_FOR_WINDOWS
	=  Tilia_from_WINDOWS
#else
	=  Tilia_from_UNKNOWN
#endif		
};



// ============================= Format of Tilia files ===========================
// Every Tilia file begins with header information that has the following format.
// You can create a Tilia file without including all of this information in the
// header.  See the documentation at the start of this file.
//
// In the future, the header may be extended to contain more information, so
// H may increase.  Do not attempt to add your own extensions to this header
// definition, since this will interfere with future header formats.
// If you need to include additional information in Tilia files, do this by
// defining a new kind of Tilia node (see VINFO_node  defintion for an example).
//
//
// The format of Tilia files on disk is:
//	Header  [starts at byte 0 in the data fork]
//		version number for format of Tilia nodes		(2 bytes)
//		H (number of header bytes - 4)					(2 bytes)
//		number of voices								(2 bytes)
//		page number; 0 is a special value; see
//		comments on MUST_SET_PERFORM definition below	(2 bytes)
//		N (number of notation contexts)					(2 bytes)
//		current editing notation context				(2 bytes)
//		T (total number of bytes in N options records)  (2 bytes)
//		OPTIONS RECORDs for N notation contexts			(T bytes; Record defn below)
//		R (number of bytes in performance record)		(2 bytes)
//		PERFORMANCE (INFORMATION) RECORD				(R bytes; Record defn below)
//		P (total number of bytes in N print records)	(2 bytes)
//		PRINT RECORDs for N notation contexts			(P bytes; Record defn in toolbox)
//		S (number of bytes in ncontext names record)	(2 bytes)
//		NOTATION CONTEXT NAMES RECORD					(S bytes, S/N bytes per name)
//	Music Binary  [starts at byte H + 4 (== T + R + P + S + 20)]
//		first Tilia node for first voice				(TiliaBytes bytes)
//		second Tilia node for first voice				(TiliaBytes bytes)
//		...												(a multiple of TiliaBytes bytes)
//		last Tilia node fir first voice					(TiliaBytes bytes)
//		first Tilia node for second voice (if any)		(TiliaBytes bytes)
//		...												(a multiple of TiliaBytes bytes)
//		last Tilia node for second voice (if any)		(TiliaBytes bytes)
//		first Tilia node for third voice (if any)		(TiliaBytes bytes)
//		...												(a multiple of TiliaBytes bytes)
//		...												(a multiple of TiliaBytes bytes)
//		last Tilia node of last voice					(TiliaBytes bytes)
//
//


//					   --- Tilia File Header: MUST_SET_PERFORM ---
// If the page number specified in the Tilia header equals "MUST_SET_PERFORM", this
// indicates that performance fields of the NOTE node (such as xlength, xttn,
// and xpitch) should be set to default values by Lime.  Lime computes the values
// from the notated duration and pitch using the same algorithms as the Recompute
// Playback menu option.
//
// If there are several lines of lyrics at the same location, they are moved apart
// so they do not overlap.  (This simplifies the OPAL to Tilia Translator program.)
//
// Lime will start editing page 1 after modifying the Tilia representation.
//
enum{
	MIN_PAGE_SIZE		= 1,		// minimum logical page size (in floating inches)
	MAX_PAGE_SIZE		= 4000,		// maximum logical page size (in floating inches);
	//								// this comes from 72 dots per inch and BIG32 as max x or y
	MUST_SET_PERFORM	= 0
};

//					  --- Tilia File Header: OPTIONS RECORD ---
// One options record is kept for each notation context in the current file format.
// This is unlike the perform record (defined below); only one perform record per piece.
//
typedef struct {
	XYPOS	page_x;					// logical page size in x (in .1 inches)
	XYPOS	page_y;					// logical page size in y (in .1 inches)
	INT8	hide_all_bar_nums;		// TRUE avoids printing any barline numbers
	INT8	hide_init_bar_num;		// TRUE avoids printing barline number at start of piece
	INT16	first_bar_num;			// first measure in piece has this number
	INT8	tickertape;				// Non-zero if this is a tickertape/autocue context;
									//	> 0 Horizontal ticker tape, pageless and systemless   scrolling;
									//	< 0 Vertical   autocue,     pageless but system based scrolling;
	//								// (cannot be printed, and cannot be edited by pre 9.15.n Lime).
	INT8	clef;					// TRUE -> make clefs automatically at start of system
	INT8	master_voice;			// Voice number + 1 of the voice to use as the master for system alignment;
	//								// 0 means force BAR synchronization.
	INT8	keysig;					// TRUE -> make keysigs automatically at start of system
	INT8	unused_8c;
	INT8	print_tuplets;			// >0 print all tuplet numbers, <0 print complete complete numbers
	REAL32	nv_spacing;				// "linearity" of chord spacing: 1 means all note
									// values get the same space, .1 means longer notes
									// get much much more space than shorter notes
	XYPOS	odd_left_points;		// points to left of staff lines, odd pages (tenth point units)
	XYPOS	first_left_points;		// points to left of staff lines in first system
	XYPOS	odd_right_points;		// points to right of staff lines, odd pages (tenth point units)
	XYPOS	last_right_points;		// points to right of staff lines in last system
	INT8	unused_8d;
	INT8	print_page_nums;		// TRUE if page numbers should be printed
	INT16	first_page_num;			// page number for first pagee in the piece
	union{
		INT16	xxxsystem_bracket;	// obsolete -- replaced by xbracket field in ENDSYST node
		INT16	base_ncontext;		// now temporary while Lime runs - in context management
	};
	UNINT16	default_between_staff;	// Default distance between staves	(points)
	UNINT16	default_between_sys;	// Default distance between systems	(points)
	INT16	default_separator_space;// Default additional distance between systems if separators	(points)
	XYPOS	above_points;			// points above top staff line (tenth point units)
	XYPOS	first_above_points;		// points above top staff line on first page
	XYPOS	below_points;			// points below bottom staff line (tenth point units)
	XYPOS	first_below_points;		// points below bottom staff line on first page
	INT8	unused_8e;
	INT8	linear_space;			// TRUE to use linear spacing; FALSE for normal spacing
	INT8	unused_8f;
	INT8	print_piece_name;		// TRUE to print the piece name at bottom of page
	INT16	unused_16;
	INT16	three_adjacent;			// obsolete (TRUE if 3 adjacent notes on piano window should create a rest! now a preference)
	INT16	XXXzoomSF;				// obsolete -- scale factor for editing zoom; see REAL32 vers below	  9.11
	INBFLD	bespoke				: 6;// (reserved for window dressing)
	INBFLD	besizable			: 2;// Music window zoom changes with horizontal width of window (fit width).
	
	INBFLD	preferred_music_font: 4;// preferred Music Font (0 means default)
	UNBFLD	metro_leadinAtStart	: 1;// Always default to a metronome lead-in when hearing from start
	UNBFLD	slurs_attachment	: 3;// Slurs preferred attachment (STEMDIR, VOICE or VOICE_SDIR))
	
	INT16	unused_16b;				 
	INT8	unused_8g;
	INT8	print_color;			// TRUE to print color noteheads on paper
	INT8	unused_8h;
	INT8	sys_sep;				// TRUE if system separation marks are to be drawn
	INT8	unused_8i;
	INT8	noNoteChanges;			// TRUE if changing notes is disabled; annotations changes ok
	UNINT8	minlwpr;				// minimum default line width for printing	(0 = minlw/2) [reserved]
	UNINT8	minlw;					// minimum default line width for screen
	XYPOS	even_left_points;		// points to left  of staff lines, even pages (tenth point units)
	XYPOS	even_right_points;		// points to right of staff lines, even pages (tenth point units)
	REAL32	zoomSF;					// scale factor for editing zoom; STD_SF means no zoom	 9.11REAL32
	UNINT16 lockedKey;				// If non-zero, prevent changes to music (allowing only markup)
	//								// Value is key required to unlock; value 1 means no key needed.
	// If you add or change  elements here, remember to adjust sizes string below.
} options_struct;
//
// This string specifies the number of bytes for each member of options_struct,
// needed for endian conversion.
//
#define options_struct_sizes "xx1121111114xxxx1122222xxxx1111222(62)(413)211111111xx42"

enum{
	// Defines how Slurs work on staves with multi-directional stems
	//
	SlursPrefer_UNSPECIFIED	= 0,	// Default as defined in options
	SlursPrefer_STEMDIR		= 1,	// Slurs stick to stem direction ignoring voice - as in 9.14 and earlier
	SlursPrefer_VOICE		= 2,	// Slurs stick to voice as far as possible
	SlursPrefer_VOICE_SDIR	= 3,	// Slurs stick to voice unless there is an existing slur in its stem direction
	SlursPrefer_DEFAULT		= SlursPrefer_STEMDIR	// if unspecified (for backwards compatibility)
};

enum{
	// The 'systems' concatenated in a ticker-tape portal page comprise the
	// the system before the reference one, the reference one and then the rest.
	//
	MAX_Ticker_Tape_Systems	= 3,	// Maximum number; may change to 4.
	MIN_Ticker_Tape_Systems	= 3,	// Minimum number (1 before and 1 after); may change to 2 (just 1 after).
	DEF_Ticker_Tape_Systems	= 3		// Default value.
};




//			--- Tilia File Header: PERFORMANCE (INFORMATION) RECORD ---
// Only one 'perform 'record is kept per Tilia file; this is unlike the options record,
// since each notation context has its own options record.  A lot of entries in the
// record have to with playback parameters, which is why it was called 'perfom'.  Nowadays,
// it contains information pertaining to the whole file, including performance.
//
enum{
	performFlags_systemMask	= 0xF,	// To extract the creation machine value from header flags

	USER_COLORS				= 16	// number of user-defined colors in preferences file
};

typedef struct {
	INT16 stacclength;		// percent of written note value played for staccato; 0-200%
	INT16 tenutolength;		// percent of written note value played for tenuto; 0-200%
	INT16 normlength;		// percent of written note value played for normal note; 0-200%
	INT8  notused1;			// Do not use before General_Relativity+1
	INT8  grace_onbeat;		// Default value for whether grace notes sound on beat or before beat
	//						// >0 if grace notes sound fully on the beat;
	//						// <0 if grace notes sound partially on and partially off the beat;
							//  0 if they sound fully before the beat.
	INT16 grace_length;		// Grace note length, as a percent of 16th note length.
	//						// Values range from 0 to 400 (400 equals a quarter note)
	
	// The following fields control the amount of change effected by the
	// commands in the Performance menu.
	//
	INT8  notused2;			// Do not use before General_Relativity+1
	UNINT8 delta_volume;	// volume change (increment/decrement for volumes  between 0..200)
	INT16 delta_pitch;		// pitch change (half-cents)
	INT16 delta_length;		// length change as a percent of current played length
	INT16 delta_start;		// start-time change, as a percent of a 16th note

	INT8  header_flags;		// Includes creating system type (see above).
	
	INT8  no_rhythm_checks;	// True to omit rhythm error checks when reading
	
	INT16 obsolete[12];		// obsolete tuning table: half-cents offset for each mod12
	//						// pitch in the scale; now tuning_table[][] is used instead;
	//						// reusable after GENERAL_RELATIVITY_VERSION for fields
	//						// whose default value is zero.
	INT16 noeffect_lth;		// TRUE if staccato, tenuto, slurs have no effect on played note attributes;
	//						// in this case stacclength, tenutolength and normlength are irrelevant.
	INT16 dynamic_level[10];// default level used for dynamic markings pppp ... ffff
	INT16 noeffect_dyn;		// TRUE if dynamic markings have no effect on volume levels;
	//						// in this case, dynamic_level[] is irrelevant.
	INT16 slurlength;		// percent of written note value played for slurs; 0-200%
	REAL32 tuning_table[7][6];
	//						// cents offset for each notated pitch in the octave;
	//						// first index is mod7 note in octave (0=c, 1=d, ... 6=b)
	//						// second index is xacc (1=--, 2=-, 3=n, 4=+, 5=++)
	RGBColor ucolor[USER_COLORS];	// copy of user_color[] array {UNINT16 red, green, blue}
	// 334 bytes to here
	UNINT16	font_map_length;// Length of following font mapping table
	UNINT16	spare_length1;	// reserved for future sub-table.
	UNINT16	spare_length2;	// reserved for future sub-table.
} perform_struct;
//
// This string specifies the number of bytes for each member of options_struct,
// needed for endian conversion.
//
#define perform_struct_sizes "22211211222112222222222222222222222222"\
"444444444444444444444444444444444444444444222222222222222222222222222222222222222222222222222"

// Following the perform structure, there may be one or more order dependent tables (records).
// These tables are included in the length of the PERFORMANCE record (simplifies compatibility).
// The order is:
//
//	FONTMAP SUB RECORDs			(font_map_length bytes)
//

//					 --- Tilia File Header: FONTMAP SUB RECORD ---
// Following the perform data, is the (optional) font mapping table.  This contains 1 or more
// 'sub-records' identifying the mapping between a font name and the Tilia identifier used
// in TPARAM nodes, etc.  In Lime 9.17, these replace the old, temporary 'Font' annotations.
// The length of the SUB RECORD is dynamic, determined from its first byte.  To avoid
// endedness conversion issues, the id is explicitly stored as two bytes
//
typedef struct{
	UNINT8	length;			// Length of the record (7 .. 255), excluding this byte.
	INT8	uniflags;		// Unicode support flags: 0=don't know; 1 does support, -1 does not support unicode.
	INT8	spare1;
	INT8	spare2;
	INT8	tilia_id_top;	// Mapping ID used in the Lime Tilia files top 8 bits
	UNINT8	tilia_id_bot;	//										bottom 8 bits.
	char	font_name[2];	// Font name (length-5 characters, including trailing 0).
} fontmap_entry_struct;


//					  --- Tilia File Header: PRINT RECORD ---
// One LimePrintRecord is kept for each notation context in the current file format.
// This is unlike the perform record (defined above); there is only one perform
// record per piece.
//
typedef struct
{
	TPrint tprint;			// obsolete pre-carbon print record; here for easy backwards compatability; 120 bytes long
	INT32 percent_scale;	// percent reduction/enlargement
	INT32 landscape;		// true for landscape paper orientation
	INT32 paper_top;		// physical paper rectangle, in points!
	INT32 paper_left;
	INT32 paper_bottom;
	INT32 paper_right;
} LimePrintRecord;
#define print_struct_sizes "22222222222222211222222222222111111222114421111111111111111111111111111111111111111444444"	// A gross of bytes
//	shorts in struct:		123456789012345  678901234567      890    1  *          #
//	bytes up to *:  31*2 + 2*4 + 12*1 = 82.  32 empty bytes (to 114), then 6 bytes (to 120) at # to fill out TPrint.


// ========================= various constant definitions =========================
//
enum{
	// ------------------------- notehead parentheses ----------------------------
	// These constants are used for parentheses around noteheads.
	//	
	RIGHT		= 1,	// parenthesis to right of notehead
	LEFT		= 2,	// parenthesis to left of notehead
	BOTH		= 3,	// parentheses on both sides of notehead
	
	// ------------------------- page related ----------------------------
	//
	ODD			= 1,
	EVEN		= 2,
	TOP			= 1,
	BOTTOM		= 2,
	
	// ----------------------------- stem directions -----------------------------
	// These constants are used for stem directions, slur directions etc.
	//
	DOWN		= 1,	// value to indicate stem/slur down
	UP			= 2,	// value to indicate stem/slur up
	UPORDOWN	= 3,	// value before the stem/slur direction has been  decided on.
	
	
	NONE		= 0 	// value to indicate none of the above
};
#define NOTDIR(d) (((d) == UPORDOWN) ? UPORDOWN : UPORDOWN - (d))


// ------------------------------- note values -------------------------------
// Note values have 16 integer bits, 16 fraction.
//
enum{
	SIXTEENTH	= 0x01000L,	// value of a sixteenth note (semi-quaver)
	EIGHTH		= 0x02000L,	// value of a quarter note	 (quaver)
	QUARTER		= 0x04000L,	// value of a quarter note	 (crotchet)
	HALF		= 0x08000L,	// value of a half note		 (minim)
	WHOLE		= 0x10000L,	// value of a whole note	 (semi-breve)
	BREVE		= 0x20000L,	// value of a breve note (a double-whole note)
	DBREVE		= 0x30000L,	// value of a dotted breve note
	DDBREVE		= 0x38000L,	// value of a double dotted breve
	DDDBREVE	= 0x3C000L	// value of a triple dotted breve
};

// ------------------------------- coordinates --------------------------------
// Constants to convert from internal x,y representation to screen coordinates.
// Note that, currently, screen resolution is only perfect at zoom factors that
// are exact multiples of the normal (2x, 3x, ..., 10x). 
//
// On a Macintosh there is an absolute (QuickDraw) limit to the zoom possible
// on the screen of 8191 pixels in the X direction, beyond that it will crash.
// The maximum zoom is, therefore, score dependent.			// JA20111128 4.1.11b
//
#define STD_SF		  10.0						// Tilia's x,y locations are kept to tenth point unit accuracy,
#define STD_SF_int	  10						// at standard no-zoom display size.

#define MAXIMUM_SF		(REAL32)1.0				// Absolute Maximum zoom
#define MINIMUM_SF		(REAL32)1000.0			// 1%; only usable as a place holder
#define STANDARDSIZE_SF	(REAL32)STD_SF			// curSF value for standard internal zoom 
#define ENLARGE_SF		(REAL32)(STD_SF/1.5)	// curSF value for enlarged zoom setting (1.5x)

#define BIGSIZE_SF		(REAL32)(STD_SF/2.0)	// curSF value for 'big' zoom setting
#define FULLSIZE_SF		MAXIMUM_SF				// curSF value for full size
#define HUGESIZE_SF		(REAL32)(STD_SF/3.0)	// curSF value for 'Huge size" zoom setting
#define LARGESTSIZE_SF	(REAL32)(STD_SF/4.0)	// curSF value for largest zoom available for normal users
#define NORMALSIZE_SF	ENLARGE_SF				// curSF value for 'normal zoom setting (actually enlarged 1.5x)
#define SMALLSIZE_SF	STANDARDSIZE_SF			// curSF value for 'Small' zoom setting
#define SUPERSIZE_SF	(REAL32)(STD_SF/8.0)	// curSF value for 8x
#define TINYSIZE_SF		(REAL32)(STD_SF/0.5)	// curSF value for 'Tiny'  zoom setting

#define GENERIC_FITWIDTH_SF (REAL32)-1.0		// Generic value for flagging resizable fit-width
//												// Currently 0 also accepted for compatibility 
#define GENERIC_DEFAULT_SF	(REAL32)0.0			// Use appropriate default (currently only if resizable set)

// Coordinate indicating "continue at current location".
// Lime does not internally generate these, except in Graphics Elements
// because it makes Allow Any Option-click processing impossibly messy;
// [the value -1111 used to be used by the old OPAL->Lime translator].
//
enum{ ADJACENT = UNDEFXY };



// --------------------------- Tilia font numbers --------------------------------
//
// Certain fonts numbers are remapped in the Tilia representation to ensure
// consistency among different Macs.  (Fonts will have different font numbers on
// different Macs.)
//
// Hidden text nodes provide the font names for other font numbers used by TPARAM nodes.
//
enum{
	TILIA_0			=  0,
	TILIA_MARL		= -1,	// special Tilia font number for Marl font
	TILIA_TUFA		= -2,	// special Tilia font number for Tufa font
	TILIA_SONATA	= -3,	// special Tilia font number for Sonata font
	TILIA_ARIAL		= -4,	// special Tilia font number for Arial font
	TILIA_GENEVA	= TILIA_ARIAL,
	TILIA_TIMES		= -5,	// special Tilia font number for Times font
	TILIA_SYMBOL	= -6,	// special Tilia font number for Symbol font
	TILIA_SIMPLE	= -7,	// special Tilia font number for Simplified notation font (reserved)
	TILIA_MIN_FONT	= -15	// Minimum special Tilia font
};

// Determine if a given Tilia font number is a music font.
//
#undef  IS_MUSIC_FONT
#define IS_MUSIC_FONT(tilfont) ((tilfont) >= TILIA_SONATA && (tilfont) <= TILIA_MARL)


// ------------------------ character mapping typedefs ----------------------------
//

typedef struct {			// struct for the information needed for character mapping
	INT16		font;		// font number; certain font numbers remapped from standard
	//						// Mac numbers: TILIA_MARL, TILIA_TUFA, TILIA_SONATA,
	//						// TILIA_ARIAL, TILIA_TIMES, TILIA_SYMBOL, etc
	UNINT8		charset;	// Windows charset number (for foreign languages)
	UNINT8		size;		// point size
	TiliaStyle	face;		// bold, italic, underline, etc
	char		ch;			// the character
	XYPOS		xoff, yoff;	// x and y offset for centering the character
} CHAR_SPEC;

enum{
	STR_MAX = 8				// maximum string length in STR_SPEC
};
typedef struct {			// struct for the information needed for short character strings;
							// this is not used by the Lime program yet
	INT16		font;		// font id
	char		str[STR_MAX];// the character string, teminated with '\0'
	char		str0;		// 0
	UNINT8		charset;	// (reserved) Windows charset number (for foreign languages)
	UNINT8		size;		// (reserved) point size (if non-zero)
	TiliaStyle	face;		// (reserved) bold, italic, underline, etc
} STR_SPEC;


typedef union {			// union for storing PARAM information: information
	//					// about either a character, an integer, or a float.
	CHAR_SPEC cvalue;	// if it is a character parameter, this provides the character specification

	REAL32	  fvalue;	// if it is a floating parameter, this provides the float
	REAL64	  dvalue;	// if it is a double parameter, this provides the double

	STR_SPEC  svalue;	// if it is a string parameter, this provides the string specification

	XYPOS	  ivalue;	// if it is an integer parameter, this provides the integer;
	//					// if it is a coordinate parameter, this provides the coord
	//					// (the user always sees coordinates divided by STD_SF)
} PARAM_VALUE;
//
// These strings specify the number of bytes for each member of PARAM_VALUE.
// This information is used to reorder bytes when COMPILE_FOR_WINDOWS; Windows has
// the low-order data byte first in memory, where Macs have the high order data
// byte first.
//
#define CHAR_SPEC_sizes		"21111xx"
#define INT16_sizes			"2"		// not used by PARAM_VALUE at this time
#define REAL32_sizes		"4"
#define REAL64_sizes		"8"
#define STR_SPEC_sizes		"2111111111111"
#define XYPOS_sizes			"x"



// -------------------------------- notation contexts ---------------------------
// A notation-context is used to define the pieces parts: score, flute part, violin
// part etc.  For each notation context, there is a set of voices that appear somewhere
// in the notation-context.  For example, a violin part uses the violin voice, and may
// also use other voices (such as the flute voice, if there is a flute cue somewhere).
// By definition, a voice appears in notation context k if it there is a PRINT node with
// Xcontext == k before the first NOTE node in that voice.
//
// Each node in the binary has an xcontext field, indicating which notation contexts
// should pay attention to that node.
//
enum{
	ALL_NCONTEXTS			= 0,	// indicates node is for all notation contexts
	SCORE_NCONTEXT			= 1,	// notation context 1 is the score and always has all voices active
	FIRST_PART_NCONTEXT		= 2,	// notation context for the first part
	NOT_IN_SCORE_NCONTEXT	= -1,	// all notation contexts except SCORE_NCONTEXT
	NCONTEXT_NOT_SELECTED	= -128	// no contexts (used internally)
};


// --------------------------- varieties of annotations ------------------------------
// When a user modifies a Lime graphic using the Allow Any Option-click menu option,
// Lime generates LINE or TPARAM/TTEXT nodes which contain the modified graphic.  Each
// time Lime reformats the page, it must identify the graphic and substitute the modified
// version.  This identification is done using the variety (defined in the enum below), and
// the oridinal of this variety associated with this note.
//
// A variety 0 (FALSE) indicates "normal" annotation.  Values 1 through 3 are not used
// (reserved for indicating hidden annotations internally in Lime subroutine arguments).
enum {
	vtie = 4, vslur, vtupl, vfretn, vbarn, vpagen, vstaffln, vpianotabstaff,			//	4-11
	vsysline, vsysbtop, vsysbleft, vsysbbetw, vsysbbot, vmainbeam, vsubbeam,			// 12-18
	vstemdown, vstemup, vbeamstemdown, vbeamstemup, vpianoroll, vledgerln,				// 19-24
	vpianotabledger, vbar1, vbar2, vbar3, vbarbetw1, vbarbetw2, vbarbetw3,				// 25-31
	vtopwingright, vtopwingleft, vbotwingright, vbotwingleft, vferm, vaccent,			// 32-37
	vstacmar, vbracetop, vbraceabove, vbracemid, vbracebelow, vbracebot, vmultirestn,	// 38-44
	vnhead, vrest, vdurdot, vacc, vupflag, vdnflag, vtrem, vclef, vtime, vcut,			// 45-54
	vcommon, vrepeatleft, vrepeatright, vkey, vhairpin1, vhairpin2, vtupbracknum,		// 55-61
	vtupbrackleftvert, vtupbrackrightvert, vtupbracklefthoriz, vtupbrackrighthoriz,		// 62-65
	vnotebrackleftvert, vnotebrackrightvert, vnotebrackhoriz, vfretl, vfing,			// 66-70
	vbend, vparen, vgliss, vbendrange, vslash, vsyssep, vrectvert1, vrectvert2,			// 71-78
	vrecthoriz1, vrecthoriz2,															// 79-
	
	variety_TOTAL, variety_TEMPORARY=255
};



// --------------------- Typedefs for various kinds of Tilia nodes ---------------------
// Type definitions for the various kinds of music binary nodes.
// All Tilia nodes are exactly TiliaBytes bytes long.  This constant size is used for memory
// allocation purposes; some bytes are unused for certain node kinds.
// The first two bytes are common to all node kinds:
//	  - The first byte is Xkind, the kind of the node. (See the list of node kinds below.)
//	  - The second byte is Xcontext, indicating which notation contexts this node
//		is relevant to. Nodes with Xcontext==ALL_NCONTEXTS apply in all notation contexts.
//		Nodes with Xcontext==NOT_IN_SCORE_NCONTEXT apply in all notation contexts except
//		the score.  Other nodes apply only in the notation context whose number matches
//		the Xcontext value; the Xcontext value SCORE_NCONTEXT is reserved for the score.
//
// For internal Lime processing, the Tilia nodes are represented as a linked list;
// a 4-byte Xnext and a 4-byte Xback pointer are added to each node; in addition
// there is a 4-byte control structure.  Internally to Lime this brings the size of
// Tilia nodes up to 64 bytes.  In Lime, a LINKS macro definition precedes "#include Tilia.h";
// the LINKS macro creates the Xnext, Xback pointers and the 4-byte internal control structure.
// By default the following code leaves the LINKS macro defined to be null.
//

#define DATA_BYTES_IN_A_TILIA_NODE (TiliaBytesInMemory - 2)
#ifndef LINKS
	#define LINKS
	typedef void*	DefTiliaPtr;
#else
	typedef union	TEMPTilia*	DefTiliaPtr;	// forward version of TiliaPtr type
#endif

//								--- node kinds ---
// The standard Tilia definition uses the node kinds listed below.  Many of
// these are node kinds which are stored on disk.  Lime uses additional node kinds
// internally that are never stored on disk; those are labeled "Temp Node" here.
// Nodes labeled GE here are graphics element nodes, whose xcontext field must be 0.
//
enum{
	FREE			= 75,		// unused node (on free list)
	NEST			= 76,		// nested tuplet
	CDDATA3			= 77,		// Temp Node: Further extension to chord data (7.5.311. obsolete not needed any more)
	BARSEP			= 78,		// companion to BAR node: indicates no bar drawn from staff to staff+1
	XMLT			= 79,		// Temp Node: For MusicXML import: timing from MusicXML file
	NIFFT			= XMLT,
	COPY			= 80,		// Temp Node: For copy-paste operation: separates voices in linked list
	STMOD			= 81,		// Tie and Slur modification info
	NIFFX			= 82,		// X-location from NIFF import from scanning software
	xxxFNAME		= 83,		// obsolete: font name associated with a font number; now with hidden text
	VOICES			= 84,		// Temp Node: For Voices... dialog: info on voices which print together
	NOTEPTR			= 85,		// Temp Node: pointer to assoicated note, used in GE list
	PLAY			= 86,		// Temp Node: For time-ordered linked lists of Hear option PLAY nodes
	ROD				= 87,		// Temp Node: For linked list of rods in relocation pass
	SPRING			= 88,		// Temp Node: For linked list of springs in relocation pass
	UNDO			= 89,		// Temp Node: For linked list of Undo and Redo changes
	MARK			= 90,		// Temp Node: Separates Undo and Redo changes
	COLOR			= 91,		// color for notehead
	GE_SPARE		= 92,		// Graphics Element: unused/reserved(must have 0 xcontext field)
	GE_CHAR			= 93,		// Graphics Element: character		(must have 0 xcontext field)
	GE_STRING		= 94,		// Graphics Element: text string	(must have 0 xcontext field)
	GE_LINE			= 95,		// Graphics Element: line			(must have 0 xcontext field)
	GE_CURVE		= 96,		// Graphics Element: curve			(must have 0 xcontext field)
	RMOVE			= 97,		// move rest up or down
	SPACC			= 98,		// extra or special accidentals for note
	AMOVE			= 99,		// move accidental right or left
	VINFO			= 100,		// voice information
	NOTE			= 101,		// note or rest
	CLEF			= 102,		// clef
	KEYSIG			= 103,		// key signature
	TIMESIG			= 104,		// time signature
	BAR				= 105,		// bar line
	DISABLED		= 106,		// Temp Node: temporarily disabled ENDSYST, CLEF & KEYSIG nodes
	TPARAM			= 107,		// parameters for text: location, style, font
	TTEXT			= 108,		// text
	LINE			= 109,		// lines or curves
	PARAM			= 110,		// parameter set: character mapping and formatting constants
	TEMPO			= 111,		// Obsolete: Tempos are now specified by annotations
	NTDATA			= 112,		// Temp Node: Note data node (temporary extension to NOTE node)
	CDDATA			= 113,		// Temp Node: Chord data node (temporarily after last note in chord)
	CDDATA2			= 114,		// Temp Node: Extension to chord data node
	ENDSYST			= 115,		// end of system / end of page / end of piece
	PRINT			= 116,		// print formatting node
	SDRAG			= 117,		// drag the staff up or down
	NA				= 118,		// NA stands for "not applicable".	This is a fake node kind
					 			// that is used internally to Lime to indicate a mismatch
					 			// of notation context (based on the Xcontext value).
	BEAMING			= 119,		// beaming values
	PDATA			= 120,		// Temp Node: Position-data node
	SPACE			= 121,		// blank space -- supported only for backward compatability
	MIDI			= 122,		// Temp Node: Used in Record menu option; raw MIDI data
	CMOVE			= 123,		// move the chord right or left; affects no other chords
	ZONE			= 124,		// this chord is in a new relocation zone; specifies new
					 			// location for the chord, and relocates chords before and
					 			// after appropriately
	NTVIEW			= 125,		// Temp Node: Note data for another view
	PVIEW			= 126,		// Temp Node: position-data for another view
	RNOTE			= 127,		// Temp Node: Used in Record and Import MIDI menu options

	FIRST_Tilia_KIND= FREE,		// see also Tilia_kind_names[], Tilia_member_sizes[],
								// and the Tilia union definition in LnkTilia.h.
	LAST_Tilia_KIND	= RNOTE,	// 

	FIRST_GE		= GE_SPARE,	// first GE type (GE types must have 0 xcontext)
	LAST_GE			= GE_CURVE,	// last GE type	 (GE types must have 0 xcontext)

	MAX_Tilia_KIND	= 127		// do not use over 127, without checking for use in INT8 values
};


//								--- node definitions ---
// There is one "struct" definition for each Tilia node kind.  Each struct definition
// begins with an Xkind and Xcontext field, as described above.
//
// In Lime it is convenient to include all of these "struct" definitions as
// part of a large "union" definition.
//
// Each struct also has a "sizes" string which specifies the number of bytes for
// each member of the struct, and the bitfield widths for any bitfield definitions
// in the struct. This information is used to reorder bytes when COMPILE_FOR_WINDOWS.
// Windows has the low-order data byte first in memory, where Macs have the high order
// data byte first.  Windows also assigns bit fields right to left in a byte, where Macs
// assign them left to right.
//
// ******************************************************************************
// *																			*
// * It is absolutely crucial that every structure subject to such 'endian'		*
// * conversion is properly aligned with its "sizes" string. Also, because many *
// * compilers have a habit of aligning multi-byte fields to 2 or 4 byte		*
// * boundaries, unused bytes should be inserted to pad to the required byte	*
// * alignment to accommodate them.												*
// *																			*
// ******************************************************************************
//

//								=== VINFO ===
// The VINFO node contains information about the voice, such as the part name.  It
// should follow the ENDSYST node that starts the voice (theone that has xstartpiece==TRUE).
//
// In and after UNICODE_ANNOTATION_VERSION for Lime files, a VINFO node may be
// followed by TTEXT nodes containing the full part name in UTF8.
//
enum{
	MAXPNAME		= 16,		// max number of characters in old part name (including '\0')
	MAXVINFONAME	= 24		// Max chars (excluding '\0') in VINFO part name in UNICODE_ANNOTATION_VERSION files and later
};
enum{
	hearme_DEFAULT	= 0,		// Xhearme usually defaults to yes, Xhearme2 to Xhearme
	hearme_NO		= 1,	
	hearme_YES		= 2,
	hearme_MAYBE	= 3			// reserved, currently just means Yes
};
enum{
	synth_default	= 0,		// default synthesizer; use only basic MIDI features
	synth_fb01		= 1,		// Yamaha FB01 with fractional pitch capabilities
	synth_mono_pbend= 2,		// Use mono mode and pitch bend for fractional pitches
	synth_silent	= 3,		// No MIDI output for this voice
};

//								== VINFO structure ==
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is VINFO
	INT8	Xcontext;			// defines the notation contexts this node applies to

	char	Xpname[16];			// Old Part name[MAXPNAME], including the terminating '\0';
	//							// for downwards compatibility post UNICODE_ANNOTATION_VERSION.
	//							// If 1st byte is zero, Xvinfoname (abbreviated part name) should be used.
	//							// After UNICODE_ANNOTATION_VERSION, 1st byte will ALWAYS be 0.
	//							// but remaining bytes will be free to use.
	
	UNBFLD	mmods		: 4;	// Reserved for multiple MIDI output devices
	UNBFLD	Xmidichan	: 4;	// MIDI channel number (0-15) for the voice
	//							// (4 bits guarantee channel cannot be out of range).
	
	UNBFLD	Xhearme		: 2;	// status of this voice in the Hear dialog
	UNBFLD	Xhearme2	: 2;	// status of this voice for auto hearing
	UNBFLD	xxxxx		: 4;	// unused bits
	
	UNBFLD	yyyyy		: 4;	// unused bits
	UNBFLD	Xmidisynth	: 4;	// type of MIDI synthesizer in used; see definitions above
	
	UNBFLD	zzzzz		: 4;	// unused bits
	UNBFLD	Xmonochans	: 4;	// If synth_mono_pbend, number of MIDI channels to use
	
	INT8	XvoiceOnPart;		// 0 for first voice on part, N-1 for further voices
	UNINT8	Xvinstrument;		// Default Midi instrument + 1 if non-zero
	INT16	spare2;
	// 26 bytes to here
	char	Xvinfoname[24];		// UTF8 Part name[MAXVINFONAME] (abbreviated part name)	(UNICODE_ANNOTATION_VERSION)
	char	Xtext0;				// Always zero to guarantee to terminate Xvinfoname.  N.B. MUST be same place as in TEXT node;
	//							// (and should be forced to zero if node is read from disk).
	UNINT8	spare1;				// reserved for part name related information (may indicate following TTEXT node)
} VINFO_node;
#define VINFO_sizes "111111111111111111(44)(224)(44)(44)112"	// All following fields are single bytes.



//								=== NOTE ===
// The NOTE struct defines all of the data that is stored about a single note or rest.
// You can look at the values of many of these fields in an existing Tilia file by using the
// "Internal Fields" menu item when you run Lime.
//
enum{								// xmod7 note values
	mod7_C				= 0,
	mod7_D				= 1,
	mod7_E				= 2,
	mod7_F				= 3,
	mod7_G				= 4,
	mod7_A				= 5,
	mod7_B				= 6,
	MAXMOD7				= 72		// highest value for xmod7
};
enum{								// Xlgval values for note lengths: 
	LGB					= 0,		// breve note (a double-whole note)
	LGW					= 1,		// whole note
	LGH					= 2,		// half note
	LGQ					= 3,		// quarter note
	LGE					= 4,		// eighth note
	LGS					= 5,		// sixteenth note
	LGTS				= 6,		// thirtysecond note
	LGSF				= 7,		// sixtyfourth note
	LGOTE				= 8,		// 128th note
	LGTFS				= 9,		// 256th note	// not yet supported
	MINLGVAL			= 0,
	MAXLGVAL			= 8,
	MAXLGVALUE			= 14,		// Absolute maximum that could ever be supported	
	BADLGVALUE			= 15,		// Illegal Value (used internally)	
	MAXTEMPOLGVAL		= 6,		// thirtysecond note is fastest value for tempo marking

	MAXDOTS				= 3			// triple-dotted is the maximum dotted note value	
};
enum{								// Xacc accidental values
	DOUBLEFLAT			= 1,
	FLAT				= 2,
	NATURAL				= 3,
	SHARP				= 4,
	DOUBLESHARP			= 5
};
enum{
	TREMMAX				= 3,		// Maximum number of tremolo bars in Xtremolo
	
	ANYSTRING			= 0,		// Use default algorithm to choose string for tablature notation.
	MAXSTRING			= 7			// largest string number allowed for tablature notation	
};
enum{								// Note head values for Xnotehead
	NHNORMAL			= 0,		// normal notehead (3 types: whole, half, and filled)
	NHPERC				= 1,		// percussion notehead (2 types: whole&half, quarter&faster
	NHOHARMONIC			= 2,		// open harmonic notehead (1 type)
	NHCHARMONIC			= 3,		// closed harmonic notehead (1 type)
	NHOLOZENGE			= 4,		// open lozenge notehead (1 type)
	NHCLOZENGE			= 5,		// closed lozenge notehead (1 type)
	NHNONE				= 6,		// no notehead is plotted
	NHPIANOROLL			= 7,		// piano-roll notehead
	NHSQUARE			= 8,		// square notehead
	NH_ICON_MAX			= 8,		// maximum NH value that is selectable from icon
	NH_MAX				= 15		// maximum value for xnotehead field
};
enum{								// Tuplet definitions
	XINTIMEOF_SHIFT		= 2,		// amount by which Xintimeof values are shifted
	XINTIMEOF_MULT		= 4,		// amount by which Xintimeof values are multipled
	XINTIMEOF_MAX		= 63		// maximum value for Xintimeof field (before shift)
};
enum{								// Xpitch values
	PITCH_CENT			= 0x002,	// A cent is 1/128th of a half-step
	PITCH_HALFSTEP		= 0x100,
	PITCHSHIFT			= 8,
	PITCHMASK			= 0xFF,
	MAXPITCH			= 0x7F00
};
#define HALF_CENT(xpch) (200 * ((xpch) >> PITCHSHIFT) + ((xpch) & PITCHMASK))
enum{								// Values for xgraced
	graced_NOTGRACE		= 0,
	graced_ONBEAT		= 1,		// Sounds on the beat (semi bit significant)
	graced_BEFOREBEAT	= 2,		// Sounds before beat (semi bit significant)
	graced_OFFONBEAT	= 3,		// Sounds partially before and partially on beat
	graced_MASKBEAT		= 3,
	graced_NACHSCHLAG	= 0x80		// 0x80 = Negative for Nachschlag (top bit)
};

//								== NOTE structure ==
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is NOTE
	INT8	Xcontext;			// notation context, should always be 0, all contexts,
	//							// except in the rare case of a very temporary note, while Lime runs.

	INT8	Xmod7;				// 7*octave + notename; notename values are mod7_C,
	//							// mod7_D, mod7_E, ..., mod7_B (see definitions above)
	//							// An odd xmod7 value indicates the note is on
	//							// a staff-line in all clefs except TREBLEOCT

	UNBFLD	Xnostem	  : 1;		// TRUE if stem for note is not to be plotted
	UNBFLD	Xseparate : 1;		// TRUE to avoid combining rests together
	UNBFLD	Xlgval	  : 4;		// lg of notated note length, uses the values defined above:
	//							// LGW=0 for whole note, LGQ=3 for quarter, etc.
	
	UNBFLD	Xbstem	  : 1;		// the stem goes between staves, and this is to go on
								// the "other" staff.  This is normally set on the lowest
								// notes in a stem-down chord (to move those lowest notes to
								// the staff below and extend the stem between the staves)
								// or on the highest notes in a stem-up chord (to move those
								// highest notes to the staff above and extend the stem between
								// the staves).  The staff below (for stem-down notes) or above
								// for stem-up notes) usually has only hidden rests.  See
								// comment in placecd.c for more information.
	UNBFLD	Xtrembeam : 1;		// TRUE if the beam should be drawn to indicate a tremolo
								// between two different pitches.  This can only be used on
								// two notes which are beamed together and have a tuplet value
								// like "2 in the time of 16".  (This tuplet value indicates
								// that the two notes actually take the place of 16 notes.)
								// The notehead is changed to a longer note; in the case of
								// change to a whole note, the beams are drawn between the
								// noteheads; in the case of change to a quarter note or faster,
								// the beams are not drawn up to the stems.

	UNBFLD	Xendvcd	  : 1;		// TRUE if last note in voice chord
	UNBFLD	Xgliss	  : 1;		// glissando flag
	UNBFLD	Xtie	  : 2;		// tie: NONE, UP, DOWN, UPORDOWN; indicates a tie from
								// this note to the next -- the ending note in the tie
								// does not have this field set
	UNBFLD	Xtenu	  : 1;		// TRUE if note has tenuto mark
	UNBFLD	Xstacc	  : 1;		// TRUE if note has staccato mark
	UNBFLD	Xgrace	  : 1;		// TRUE if it is a grace note
	UNBFLD	Xnoslash  : 1;		// if note is a grace note, TRUE if no slash on grace

	UNBFLD	Xrest	  : 1;		// TRUE if it is a rest
	UNBFLD	Xhider	  : 1;		// TRUE if rest is not to be plotted
	UNBFLD	Xhidet	  : 1;		// TRUE if tuplet marks not to be plotted
	UNBFLD	Xstring	  : 3;		// For guitar tablature notation: user-specified
	//							// string for the note; value ANYSTRING indicates
	//							// user did not specify a specific string.
	UNBFLD	Xdots	  : 2;		// duration dots after a note

	UNBFLD	Xacc	  : 3;		// accidental; one of the values defined above.
	//							// This accidental may not need to be plotted;
	//							// if it is implied by the key signature, or
	//							// by a preceding accidental in the same measure,
	//							// Xaccplot==FALSE can suppress plotting it.
	//							// Note that this field needs to be set even if
	//							// an SPACC node (special accidental symbol) is present;
	//							// xacc is needed in case SPACC is removed by user.
	
	UNBFLD	Xaccplot  : 1;		// TRUE if accidental is to be plotted
	UNBFLD	Xferm	  : 1;		// TRUE if there is a fermata
	UNBFLD	Xdir	  : 2;		// stem direction  UP, DOWN, or UPORDOWN
	UNBFLD	Xcbeam	  : 1;		// TRUE if note-beaming must not end here

	UNBFLD	Xebeam	  : 1;		// TRUE if note-beaming must end here
	UNBFLD	Xcsbeam	  : 1;		// TRUE if note-subbeaming must not end here
	UNBFLD	Xesbeam	  : 1;		// TRUE if note-subbeaming must end here
	UNBFLD	Xaccent	  : 2;		// 0 no accent, 1 small accent, 2 big accent
	UNBFLD	Xmodvol	  : 1;		// for editing: indicates xvolume modified with louder/softer (obsolete)
	UNBFLD	Xselected : 1;		// used by the editor to mark selected notes for
								// GROUP_SELECT when MODE==MUSIC_MODE
	UNBFLD	Xcue	  : 1;		// TRUE if it is a cue note

	UNINT8	Xtupl;				// 1 for normal	  3 for triplets  see xintimeof field below

	UNBFLD	Xnotehead  : 4;		// notehead type; one of the values defined above
	//							// xnotehead_to_char() maps to character numbers
	
	UNBFLD	Xslur	   : 2;		// slur: NONE, UP, DOWN, UPORDOWN; indicates a slur from
	//							// this note to the next -- the ending note in the slur
	//							// does not have this field set
	
	UNBFLD	Xtremolo   : 2;		// single-pitch tremolo:
	//							// 0, 1, 2 or 3 tremolo bars drawn on the note stem

	UNINT8	Xintimeof;			// used with xtupl field: "xtupl in the time of xintimeof";
	//							// examples: "3 in the time of 2" or "5 in the time of 7";
	//							// this field is shifted left by 2 to allow things like
	//							// "5 quarters in the time of 7 eighths" expressed by
	//							// "5 in the time of 3.5"

	UNBFLD	Xdashtie   : 1;		// TRUE if tie is to be dashed; used with Xtie field
	UNBFLD	Xdashslur  : 1;		// TRUE if slur is to be dashed; used with Xslur field
	UNBFLD	Xthumb	   : 1;		// TRUE for tablature note played by thumb
	UNBFLD	Xmodp	   : 1;		// for notes: indicates xpitch modified with lower/higher;
								// for rests: indicates beaming over it is allowed if hidden
								// (used to be a pad-rest added for NIFFX alignment)
	UNBFLD	Xbendtie   : 1;		// TRUE if tie-with-pitch bend; draws a "bend" tie; in tab
								// a bend-up or bend-down symbol replaces tied-to note
	UNBFLD	Xparen	   : 2;		// parentheses at notehead: NONE, LEFT, RIGHT, or BOTH
	UNBFLD	Xbendslur  : 1;		// TRUE if slur-with-pitch bend; draws a "bend" slur; in tab
								// a bend-up or bend-down symbol replaces slurred-to note

	UNINT8	Xvolume;			// volume for playback, 0..200; user sees this
								// as a percentage of maximum 0..100 in steps of .5
	UNBFLD	Xnachschlag : 1;	// this indicates grace after the beat (only set if also xgrace is set)
	UNBFLD	Xcrawlslur	: 1;	// TRUE to move slur at stem as close to notehead as possible
	UNBFLD	Xseparateh	: 1;	// TRUE not to combine note head with another voice on same staff
	UNBFLD	unused_bit5	: 5;	// do not use until GENERAL_RELATIVITY_VERSION+1 (actually Xseparateh was pushing it)

	UNINT16 Xpitch;				// The audio pitch of the note.
								// This is an 8-bit notenumber (halfstep), followed
								// by an 8-bit fraction.
								// The 8-bit fraction is in half-cent units, and
								// takes on the range 0..200.
								// during MusicXML import: PACK_ID(partID,voiceID) for note
	INT32	Xlength;			// Playback duration: length in whole notes.
	//							// Format: 16 bit integer part, 16 bit fractional part.
	INT32	Xttn;				// Playback duration: time to next in whole notes.
	//							// Format: 16 bit integer part, 16 bit fractional part.
	
	INT8	unused1;			// reserved for something (e.g. octave-shifting pitch); do not use until GENERAL_RELATIVITY_VERSION+1
	INT8	Xgraced;			// Non-zero if some form of grace note; negative (top bit set) for Nachschlag.
	//							// reserved to take over Xgrace and Xnachschlag.  Bottom 3 bits:
	//							//	1	on-beat;
	//							//	2	not on-beat;
	//							//	3	partially on-beat.
	
	INT8	unused[10];			// Free (perhaps for NEST and STMOD contents?)
	INT8	unused2;			// do not use until GENERAL_RELATIVITY_VERSION+1
	
	INT8	Xvoice;				// temporary data while Lime runs: voice number for node
	REAL32	Xn_dur_mult;		// Cached Polytemporal duration multiplier.
	//							// If non-zero, the duration of this chord is multiplied by this value.	
	//							// It is reset from prevailing Time Signatures when a file is read.
	// 42 bytes to here
} NOTE_node;
#define NOTE_sizes "111(11411)(1121111)(11132)(31121)(1112111)1(422)1(1111121)1(1115)244111111111111114"





//								=== TIMESIG ===
//
enum{
	NORMAL_TIME		= 0,		// xtimetype value for normal time signature
	CUT_TIME		= 1,		// xtimetype value for cut time signature
	COMMON_TIME		= 2,		// xtimetype value for common time signature
	COMPOUND_TIME	= 3,		// xtimetype value to indicate Compound time and that the nominal values should be displayed
	DEFAULT_TIMESIG	= 4,		// default time signature is 4/4
	MAXIMUM_TIMESIG	= 240,		// maximum time signature (triple dotted 128th note!)
	UNKNOWN_TIMESIG	= UNDEF16	// Used internally for Xbpm to indicate undefined time signature
};

enum{							// Anything except normal or hidden is unusual:
	placeAboveStaff		= 1,	// Place above staff  at preceding bar-line 
	placeAboveSystem	= 2,	// Place above system at preceding bar-line 
	placeAboveStaffAlt	= 3,	// Place above staff  at preceding bar-line reserved for alternative representation
	placeNormalAlt		= 4,	// Place normally on staff					reserved for alternative representation
	placeBelowStaff		= 5,	// Place below staff  at preceding bar-line 
	placeBelowSystem	= 6,	// Place below system at preceding bar-line 
	placeBelowStaffAlt	= 7,	// Place below staff  at preceding bar-line reserved for alternative representation 
	placeOddly,					// maximum recognized value
	placeHidden			= -128,	// Any negative value means do not show at all (unless showing hidden symbols)
	placeNormal			= 0		// Place normally on staff
};

enum{							// what signature actually applies to a pickup measure in alternating signature:
	tsig_AltMetro_NO	= 0,	// XbpmAlternates is purely cosmetic; metronome will beat the primary rhythm;
	tsig_AltMetro_12	= 1,	// XbpmAlternates!=0:	Pick up is 1st  primary; 
	tsig_AltMetro_21	= 2,	// XbpmAlternates!=0:	Pick up is last alternate (normal); 
	tsig_AltMetro_11	= 3,	// XbpmAlternates!=0:	Pick up is an extra primary, next measure is 1st primary;
	
	tsig_AltMetro_2		= 2,	// XbpmAlternates==0:	Beat the alternate rhythm;
	tsig_AltMetro_YES	= 2		// use the alternate rhythm as per defaults.
};

enum{
	tsig_BPMALT_DEPTH	= 4		// Maximum number of consecutive measures of same signature in an alternating sequence;
	//							// There can be up to 4 of the 1st type followed by up to 3 of the 2nd (4-3),
	//							// or           up to 3 of the 1st type followed by up to 4 of the 2nd (3-4),
	//							// 4-4 is not supported.
	//							// Note that Bernstein's America is 1-1, and Dave Brubeck's Blue rondo a la Turk is 3-1.
};

// Note on Polytempo Duration Multiplier.  This is a very advanced feature to support polytemporal music,
// in which parts run at different paces through time.  The Polytempo Duration Multiplier in Xts_dur_mult
// is essentially a number which is applied to the effective note length of every NOTE following the
// time signature to change the duration and spacing.  If, for example, the multiplier is 1.5, then a
// quarter note (crotchet) will take the same time as a dotted quarter note on a part without a multiplier.
// As in most cases, there is a well defined ratio between polytemporal parts, it is possible to specify 
// and maintain the Polytemporal Multiplier as a ratio in the form of a (heavy) fraction, or a product of
// two fractions.  If more irrationality is needed, then it can be specified as a positive decimal number.
// A zero multiplier is the same as 1.0, but more efficient.

// Xts_dur_mult is defined as a FREAL32, which can either be a positive REAL32 number or can be in
// fractional form (see definition of a FREAL32).  For efficiency, the effective multiplier (REAL32) is
// cached in every NOTE in the voice following the TIMESIG node.  This cached value is written to disk;
// however it is re-cached whenever the file is read.

//								== TIMESIG structure ==
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is TIMESIG
	INT8	Xcontext;			// defines the notation contexts this node applies to

	// Xbpm and Xbeat used to be the numerator and denominator.  They are now used, semi-temporarily while Lime runs,
	// to hold the derived nominal values.  For simple time signatures they are the same as xbpmp and xbeatp.
	// For complex signatures (additive or fractional), xbpm and xbeat, they hold the nominal whole measure values.
	// The fields are, therefore, backwards compatible with pre 9.15 (GENERAL_RELATIVITY_VERSION) versions of Lime files.
	// Though written to disk, Xbpm and Xbeat are always recomputed (if Xbpmok is true).
	//
	INT16	Xbpm;				// temporary while Lime runs: nominal beats per measure (numerator)
	INT16	Xbeat;				// temporary while Lime runs: nominal beat value (denominator)

	XYPOS	Xtimex;				// temporary while Lime runs: x location of time sig
	XYPOS	Xtimeorigin;		// temporary while Lime runs: relocation x of time sig

	UNBFLD	Xtimetype2	  : 4;	// NORMAL_TIME or COMPOUND_TIME
	UNBFLD	Xtimetype	  : 4;	// NORMAL_TIME, CUT_TIME, COMMON_TIME, or COMPOUND_TIME
	//							// The xbpmp and xbeatp fields should be set, even for cut time and common time.
	UNBFLD	Xno_cwr		  : 1;	// TRUE to disable centered whole rests in these time signatures
	UNBFLD	Xbpmpok		  : 1;	// TRUE if Xbpmp, etc, have been set properly;
	//							// indicates additive and alternating fields below are valid.
	//							// if FALSE, values are derived frm Xbpm and Xbeat (i.e. converted from old file version)
	
	UNBFLD	Xpickupmetro  : 2;	// Indicates What the metronome should play.
	//							// 0 =	default (beat just the primary time).
	//							// Xbpmalternates != 0:	!=0	beat rhythms as specified - value identifies how pick-up measures are handled:
	//							//							see pickupMetro definitions above.
	//							// Not alternating:		2 = beat the secondary values; all other values = beat the primary (reserved).
	
	UNBFLD	Xbpmalternates: 4;	// non-zero if signature alternates every other bar (1-1) or more complexly:
	//							// 1 = 1-1, 2=1-2, 3=1-3, 4=1-4, 5=2-1, etc, 15=4-3
	
	// A simple time signature (used in most cases) has a single numerator (Xbpmp); it and the denominator (xbeatp)
	// will normally be the same as Xbpm and xbeat respectively.
	// An additive time signature is indicated by a non-zero Xbpma value and, optionally, Xbpmb and Xbpmf.
	// A fractional time signature is indicated by a non-zero Xbpmf but a zero Xbpma; a fraction
	// can only be 1/2 or n/4 or n/8.
	// 
	UNINT8	Xbpmp;				// primary beats per measure;
	UNINT8	Xbpma;				// non-zero for 1st additive time signature;		
	UNINT8	Xbpmb;				// non-zero for 2nd additive time signature;		
	UNINT8	Xbpmf;				// non-zero, either 3rd additive value, or fractional part (*256, i.e. 128 = 0.5)
	UNINT8	Xbeatp;				// primary beat value (time signature denominator) if Xbpmalternates
	UNINT8	unused1;			
	
	// A non-zero value of Xbpmalternates indicates alternating time signature (e.g. 6/8 3/4);
	// these fields give the alternating values
	//
	INT16	Xbpm2;				// (alternating) temporary while Lime runs: nominal beats per measure (numerator) if Xbpmalternates
	INT16	Xbeat2;				// (alternating) temporary while Lime runs: nominal beat value (denominator)      if Xbpmalternates

	UNINT8	Xbpmp2;				// (alternating) primary beats per measure;
	UNINT8	Xbpma2;				// (alternating) non-zero for 1st additive time signature;		
	UNINT8	Xbpmb2;				// (alternating) non-zero for 2nd additive time signature;		
	UNINT8	Xbpmf2;				// (alternating) non-zero, either 3rd additive value, or fractional part (*256)
	UNINT8	Xbeatp2;			// (alternating) primary beat value (time signature denominator)
	UNINT8	unused2;
	
	INT8	Xtsigplace;			// Placement of time signature
	INT8	unused3;
	
	FREAL32	Xts_dur_mult;		// Polytemporal duration multiplier (+ve decimal or fractional product);
	//							// If non-zero, the duration of all notes following this time signature 
	//							// are multiplied by this value (with appropriate rounding).
	// 40 bytes to here

} TIMESIG_node;
#define TIMESIG_sizes "1122xx(44)(1124)1111112211111111114"


//								=== CLEF ===
//
enum{
	TREBLE			= 0,		// xclef value for treble clef
	BASS			= 1,		// xclef value for bass clef
	ALTO			= 2,		// xclef value for alto clef
	TENOR			= 3,		// xclef value for tenor clef
	SOPRANO			= 4,		// xclef value for soprano clef
	TREBLEOCT		= 5,		// xclef value for treble-octave-down clef
	BARITONE		= 6,		// xclef value for baritone clef (bass clef symbol)
	PERCCLEF1		= 7,		// xclef value for single bar percussion clef
	PERCCLEF2		= 8,		// xclef value for dual bar percussion clef
	CYMBALCLEF		= 9,		// xclef value for cymbal clef
	MEZZO_SOPRANO	= 10,		// xclef value for soprano clef
	C_BARITONE		= 11		// xclef value for baritone clef (c clef symbol)
};

typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is CLEF
	INT8	Xcontext;			// defines the notation contexts this node applies to

	INT8	Xclef;				// type of clef
	INT8	Xsmallclef;			// TRUE if it is small size (clef change)
	INT8	Xhideclef;			// TRUE to skip plotting the clef
	INT8	notUsedYet;
	XYPOS	Xclefx;				// temporary data while Lime runs: clef x location
	XYPOS	Xcleforigin;		// temporary data while Lime runs: clef relocation x

} CLEF_node;
#define CLEF_sizes "111111xx"



//								=== KEYSIG ===
//
// The key signature may be any combination of sharps, flats, or naturals
// in any order.  The 'xkeynot[]' array specifies a list of pitches, and for
// each pitch, which accidental is to be plotted at that pitch.  The values
// of the 'xkeynot[]' array are defined as follows:
//	 0 flat on c	1 flat on d	   2 flat on e	  3 flat on f	 4 flat on g
//	 5 flat on a	6 flat on b	   7 sharp on c	  8 sharp on d	 9 sharp on e
//	10 sharp on f  11 sharp on g  12 sharp on a  13 sharp on b
// If 14 is added to these values, then naturals are plotted in the
// corresponding positions.  Note that sharps and flats on the same pitch don't
// always plot in the same octave in key signatures -- for example, the flat on
// f plots an octave lower than the sharp on f in treble clef.
//
// NOTE: It is important that an all-zero key signature node is C major.  This
// is used in several places if the piece has no key signature.
//

enum{ MAXKEY	= 14 };			// max number of sharps, flats, nats in key

typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is KEYSIG
	INT8	Xcontext;			// defines the notation contexts this node applies to

	INT8	Xkeycnt;			// number of sharp/flats/nats in key
	INT8	Xkeynot[MAXKEY-1];	// see comments above; last byte of array defined below
	INT8	XkeynotEND;			// last byte for Xkeynot[] array; this is defined
								// separately to defeat compiler's word alignment of
								// even-length byte arrays.
	INT8	Xtoplay;			// add this number of halfsteps to printed pitch
								// to get the playback pitch.  for example, this
								// is -2 for instruments in b-flat
	INT8	Xkey;				// type of key; order as defined below:

	INT8	Xcolorkey;			// TRUE for automatic colorizing of key signature
	
	INT64	unused8;			// do not use before Unicode Format
	
	INT8	Xkeyhide;			// TRUE to hide this key signature
	INT8	unused;
	
	INT16	unused16[2];
	INT32	unused32[2];
	//	42 bytes to here

	XYPOS	Xkeyx;				// temporary data while Lime runs: x location of key
	XYPOS	Xkeyorigin;			// temporary data while Lime runs: relocation x of key
	//							// available as temporary flags during musicXML import.
} KEYSIG_node;
#define KEYSIG_sizes "1111111111111111111111111111112244xx"

enum{
	cmajor_key		= 0,
	gmajor_key		= 1,
	csharpmajor_key	= 7,
	fmajor_key		= 8,
	cflatmajor_key	= 14,
	aminor_key		= 15,
	eminor_key		= 16,
	asharpminor_key	= 22,
	dminor_key		= 23,
	aflatminor_key	= 29,
	KEY_NAT_OFFSET	= 14
};
#define MAJORKEY(kptr)			(xkey(kptr) < 15)
#define SHARPKEY(kptr)			((xkey(kptr) % 15) <= 7)
#define CNTKEY(kptr)			((SHARPKEY(kptr) ? xkey(kptr) : xkey(kptr) - 7) % 15)
#define XKEY_VALUE(mj,num,shp)	((mj ? 0 : 15) + num + ((shp || !num) ? 0 : 7))
#define MAJOR_SHARP_KEYS \
	QQ("C Major"), QQ("G Major"), QQ("D Major"), QQ("A Major"), QQ("E Major"), \
	QQ("B Major"), QQ("F Sharp Major"), QQ("C Sharp Major")
#define MINOR_SHARP_KEYS \
	QQ("A Minor"), QQ("E Minor"), QQ("B Minor"), QQ("F Sharp Minor"), QQ("C Sharp Minor"), \
	QQ("G Sharp Minor"), QQ("D Sharp Minor"), QQ("A Sharp Minor")
#define MAJOR_FLAT_KEYS \
	QQ("C Major"), QQ("F Major"), QQ("B Flat Major"), QQ("E Flat Major"), QQ("A Flat Major"), \
	QQ("D Flat Major"), QQ("G Flat Major"), QQ("C Flat Major")
#define MINOR_FLAT_KEYS \
	QQ("A Minor"), QQ("D Minor"), QQ("G Minor"), QQ("C Minor"), QQ("F Minor"), \
	QQ("B Flat Minor"), QQ("E Flat Minor"), QQ("A Flat Minor")




//								=== TPARAM ===
// A TPARAM node specifies the location, style and font for text.
// The actual text string is contained in one or more TTEXT nodes following
// the TPARAM node.
//
// If there are several lines of text, each line of text is implemented with
// its own TPARAM node followed by TTEXT nodes.  There are no end-of-lines in
// TTEXT nodes.
//
// If there is a font change within a line, then a new TPARAM node (with xadjacent
// set) is inserted at the proper place in the TTEXT nodes; this TPARAM node is in
// turn followed by more TTEXT nodes.
//
// TPARAM and TTEXT nodes precede the NOTE node that the text is associated with.
// Note relative text annotations are relative to the location that the
// associated NOTE gets during print formatting.
//

typedef struct {
	LINKS					// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;			// kind of node; value is TPARAM
	INT8	Xcontext;		// notation context; 0 all contexts, -1 not score

	XYPOS	Xtextx;			// x offset from anchor to first, centre, or last character depending on justification; 
							// anchor is dependent on xrelativex
	XYPOS	Xtexty;			// y offset to first character from anchor
							// anchor is dependent on xrelativey
	
	INT16	Xtextfont;		// font number for text; certain font numbers remapped from
							// standard Mac numbers: TILIA_MARL, TILIA_TUFA, TILIA_SONATA,
							// TILIA_ARIAL, TILIA_TIMES, TILIA_SYMBOL; other font
							// number mappings are specified in FNAME nodes.
	UNINT8	Xcharset;		// Windows charset value for Windows foreign language support
	UNINT8	Xtextsize;		// font size (in points)
	TiliaStyle	Xtextface;	// bit significant type face (italic, bold, underline etc).

	UNBFLD	Xdrawmode	: 2;// drawing mode: ORMODE or BICMODE

	UNBFLD	Xkeep_hyphen : 1;
							// xkeep_hyphen is TRUE if trailing hyphens should stay
							// attached to text the item; FALSE indicates that the trailing
							// hyphen should be chopped off from the text and a centered
							// hyphen should be printed between this chord and the next.
	UNBFLD	Xkeep_underline : 1;
							// xkeep_underline is TRUE if leading and trailing underline
							// characters should stay attached to text the item; FALSE
							// indicates that the underline characters should be chopped
							// off from the text and a line should be printed between this
							// chord's lyric and the next.
	UNBFLD	Xno_size_adjust : 1;
							// Xno_size_adjust is TRUE if this text is the same size independent
							// of staff size changes; if Xno_size_adjust is FALSE, the text size
							// and location is adjusted (ie, on a cue staff in a different context);
							// see xtextstsize
	UNBFLD	Xpslayout	: 1;// TRUE if LineLayoutOn is to be used when printing this
							// text item in PostScript
	UNBFLD	Xspec_underline : 1;
							// special underlining useful for text with superscripts:
							// Lime draws a single underline, no superscripted underline parts.
							// This used to be called "xchordsym" and also affected
							// transposability of text (now xclassify==cl_chordsymbol).
	UNBFLD	Xbackslash	: 1;// TRUE if backslash expressions were used in typing this

	UNBFLD Xjustified	: 2;// LEFT_JUSTIFIED, CENTER_JUSTIFIED or RIGHT_JUSTIFIED.
							// Eg, if xjustified==CENTER_JUSTIFIED, the text is centered
							// around the location derived from (xtextx, xtexty).
	UNBFLD Xpagemargin	: 1;// OBSOLETE, replaced by Xrelativex
							// This was presented as "horizontal lock" to users.
							// TRUE if the text should be placed relative to the
							// page margins, FALSE if xtextx is relative to notehead x.
							// The title or sub-title of a piece usually had
							//		xpagemargin = TRUE
							//		xjustified = CENTERED.
							// This is overridden (implied set) if xfooter is set.

	UNBFLD Xboxed		: 1;// TRUE if the text should have a box drawn around it
							// (used for rehearsal marks)
	UNBFLD Xleavespace	: 1;// TRUE if horizontal space should be left for the text
							// when the music is formatted; this is used for lyrics.
							// Lyrics used to normally have the following values:
							//		xjustified = CENTERED
							//		xfromstaff = TRUE
							//		xfooter = FALSE
							//		xleavespace = TRUE
							//		xtextx = 0  (center around the notehead location)

	UNBFLD Xfromstaff	: 1;// OBSOLETE, replaced by Xrelativey
							// This was presented as "vertical lock" to users.
							// TRUE if xtexty is the offset from the center staff line;
							// FALSE if xtexty is the offset from the notehead y
							// This is overridden if xfooter is set.
	UNBFLD Xadjacent	: 1;// TRUE if the text is a font-changed continuation of the
							// previous text; only the font, size, and style information
							// fields of the TPARAM node are used if xadjacent is set.
							// This is never used together with xeocvar, since
							// xadjacent is only generated by user annotations with
							// font changes, and xremove cannot be used to hide user annos.
	UNBFLD Xeachsys		: 1;// TRUE if this was inserted using Duplicate At Each System.
							// This has no special meaning; it is only used by the editor
							// if the user uses Clear At Each System. This field is
							// provided to allow the user to undo an accident.

	UNBFLD Xfing		: 1;// This is presented as style "fingering" or "stack vertical" to the users.
							// It is used to print fingering information in a vertical stack.
	UNBFLD Xeverypart	: 1;// TRUE if this was inserted using Duplicate In Every Part.
							// If this bit is set, the annotation will only print if
							// this is the score and xcontext==SCORE_NCONTEXT, or if
							// this is not the score and xcontext==NOT_IN_SCORE_NCONTEXT
							// and this is the top staff in the current notation context; it is
							// also used by the editor if the user uses Clear In Every Part.
							// See anno_everypart_dup() in annocopy.c.
	UNBFLD Xhidden		: 1;// TRUE if the text should not be printed on paper; on the
							// screen the text appears with a dotted underline to
							// indicate hidden text.  This is useful for dynamic markings
							// or other text which is not to be printed. This should
							// not be confused with xeocvar or xdrawmode==BICMODE.
	UNBFLD Xdefault		: 1;// TRUE if this anno serves as default for xclassify value
	UNBFLD Xeachpage	: 1;// TRUE if this was inserted using Duplicate On Each Page.
							// This has no special meaning; it is only used by the editor
							// if the user uses Clear On Each Page. This field is
							// provided to allow the user to undo an accident.
	UNBFLD Xunicode		: 1;// Annotation is UTF-8 (as opposed to MacRoman)
	UNBFLD Xfooter		: 1;// OBSOLETE, replaced by Xrelativey (and Xrelativex)
							// TRUE if position is relative to bottom left of page; this value
							// overrode xfromstaff and xpagemargin
	UNBFLD Xignore		: 1;// TRUE if Lime should not interpret anno

	UNBFLD Xtext_stsize : 4;// obsolete original staff size, in points (replaced by Xtextstsize)	// JA20150320 7.5.357.
	UNBFLD Xtext_color	: 4;// color index into color table

	UNINT8 Xeocord;			// ordinal of this variety; used with xeocvar
	UNINT8 Xeocvar;			// zero if this is a regular annotation; else it is the variety
							// of Line graphic being modified by this TPARAM/TTEXT.  This
							// is for Allow Any Option-click.
	INT8 Xsuper;			// amount to superscript (positive) or subscript (negative)
	INT8 Xexpand;			// amount to expand (move right) or condense (move left)
	INT8 Xclassify;			// classify type of text
	
	UNINT8 Xtextstsize;		// original staff size, in tenths; used with xno_size_adjust field

	UNBFLD Xrelativey	: 4;// what y positioning is relative to (Y anchor)
	UNBFLD Xrelativex	: 4;// what x positioning is relative to (X anchor)	
	
	TiliaRGBA	Xtext_rgba;	// Reserved for generalized colouring
	
	INT16 Xstbl;			// for Lime internal use: string table index in NIFF export (obsolete)
	INT16 Xfontd;			// for Lime internal use: font description   in NIFF export (obsolete)
} TPARAM_node;
#define TPARAM_sizes "11xx2111(2111111)(2111111)(11111111)(44)111111(44)111122"

// Enumeration for the Xdrawmode field
enum{
	ORMODE					= 0,	// drawing mode -- Or
	BICMODE					= 1,	// drawing mode -- Bit Clear -- called "draw in white" for users
	NODRAW					= 2		// drawing mode -- Do Not Draw (deleted xeocvar anno)
};

enum{
	HIDDEN_DESC				= 30,	// hidden text: dotted underline distance below (tenth pt)
	HIDDEN_GAP				= 2,	// hidden text: dotted underlines gap length
	HIDDEN_DASH				= 1,	// hidden text: dotted underlines dash length
	HIDDEN_WIDTH			= 1,	// hidden text: dotted underlines line width
	OFFPAGE_DASH			= 5,	// text moved from offpage: dashed indicator length
	OFFPAGE_GAP				= 1,	// text moved from offpage: dashed indicator length
	
	MAXFING					= 10	// max number of numerals in fingering TTEXT node (5 is normal max)
};

// Enumeration for the Xjustified field
//
enum{
	LEFT_JUSTIFIED			= 0,
	CENTERED				= 1,	// Centered but not moveable
	RIGHT_JUSTIFIED			= 2,
	CENTER_JUSTIFIED		= 3		// Centered and moveable	
};

// Enumeration for the Xrelative fields
//
enum{
	NIL_RELATIVE			= 0,	// unspecified (should never be used)
	NOTE_RELATIVE			= 1,	// +right/up,-left/down
	PAGE_RELATIVE			= 2,	// from bottom/left	+right/up,-left/down
	PAGE_LEFT_RELATIVE		= 2,	// from left		+right,(-left!)
	PAGE_BOTTOM_RELATIVE	= 2,	// from bottom		+up,(-down!)
	PAGE_RIGHT_RELATIVE		= 3,	// from right		+right,-left
	PAGE_TOP_RELATIVE		= 3,	// from top			-down,(+up!)
	PAGE_CENTER_RELATIVE	= 4,	// from the middle point in the page	+right/up,-left/down
	STAFF_RELATIVE			= 5,	// y only: middle +up/-down
	STAFF_LEFT_RELATIVE		= 5,	// x only: from staff left	+right,-left 
	STAFF_RIGHT_RELATIVE	= 6,	// x only: from staff right	+right,-left
	MARK_RELATIVE			= 6,	// y only: relative to last expression mark placed automatically (vertically fixed)
	MARK_ABOVE_RELATIVE		= 7,	// y only: relative to last expression mark if placed above
	MARK_BELOW_RELATIVE		= 8,	// y only: relative to last expression mark if placed below
	MARK_LEFT_RELATIVE		= 7,	// x only: relative to last mark (accidental) on the left
	MARK_RIGHT_RELATIVE		= 8,	// x only: relative to last mark (e.g. dot)   on the right [not used, this may not be possible]
	BAR_RELATIVE			= 9,	// x only: +right,-left
	BAR_LEFT_RELATIVE		= 9,	// x only: +right,-left
	BAR_RIGHT_RELATIVE		= 10,	// x only: +right,-left	
	
	// Masks for identifying relativity classes.
	//
	NOTE_RELATIVExMASK		= (1<<NOTE_RELATIVE) | (1<<MARK_LEFT_RELATIVE)	 | (1<<MARK_RIGHT_RELATIVE),
	NOTE_RELATIVEyMASK		= (1<<NOTE_RELATIVE) | (1<<MARK_RELATIVE)		 | (1<<MARK_ABOVE_RELATIVE) | (1<<MARK_BELOW_RELATIVE),
	PAGE_RELATIVE_MASK		= (1<<PAGE_RELATIVE) | (1<<PAGE_CENTER_RELATIVE) | (1<<PAGE_RIGHT_RELATIVE) | (1<<PAGE_TOP_RELATIVE),
	STAFF_RELATIVExMASK		= (1<<STAFF_RELATIVE)| (1<<STAFF_LEFT_RELATIVE)  | (1<<STAFF_RIGHT_RELATIVE),
	STAFF_RELATIVEyMASK		= (1<<STAFF_RELATIVE),
	BAR_RELATIVExMASK		= (1<<BAR_RELATIVE)	 | (1<<BAR_LEFT_RELATIVE)	 | (1<<BAR_RIGHT_RELATIVE),
	
	NIL_RELATIVE_MASK		= (1<<NIL_RELATIVE)
};
#define is_PAGE_RELATIVE(r)			((1<<(r)) & PAGE_RELATIVE_MASK)
#define is_NIL_RELATIVE(r)			(!(r))
#define is_NOTExRELATIVE(r)			((1<<(r)) & NOTE_RELATIVExMASK)
#define is_NOTEyRELATIVE(r)			((1<<(r)) & NOTE_RELATIVEyMASK)
#define is_BAR_NOTExRELATIVE(r)		((1<<(r)) &(NOTE_RELATIVExMASK | BAR_RELATIVExMASK))
#define is_STAFFxRELATIVE(r)		((1<<(r)) & STAFF_RELATIVExMASK)
#define is_STAFFyRELATIVE(r)		((r) == STAFF_RELATIVE)
#define is_STAFF_NOTEyRELATIVE(r)	((1<<(r)) &(STAFF_RELATIVEyMASK| NOTE_RELATIVEyMASK))
#define is_STAFF_PAGExRELATIVE(r)	((1<<(r)) &(STAFF_RELATIVExMASK| PAGE_RELATIVE_MASK))
#define is_STAFF_PAGEyRELATIVE(r)	((1<<(r)) &(STAFF_RELATIVEyMASK| PAGE_RELATIVE_MASK))
#define is_STAFF_MARKyRELATIVE(r)	((1<<(r)) &(STAFF_RELATIVEyMASK| (1<<MARK_ABOVE_RELATIVE)  | (1<<MARK_BELOW_RELATIVE)))

// For example:
//
// The title or sub-title of a piece usually have:
//		xrelativex = PAGE_CENTER_RELATIVE
//		xrelativey = PAGE_TOP_RELATIVE
//		xjustified = CENTER_JUSTIFIED (or CENTERED).
//		xtextx     = 0  (center around the reference location)
//
// Lyrics usually have:
//		xrelativex = NOTE_RELATIVE
//		xrelativey = STAFF_RELATIVE
//		xjustified = CENTERED
//		xleavespace= TRUE
//		xtextx     = 0  (center around the notehead location)
//


// Enumeration for Xclassify field.	  JA20120228 Moved here from Lnktilia.h
//
enum {
	cl_maxTextAnnotation	= +128,	// Used internally
	cl_notTextAnnotation	= -128,	// Used internally
	cl_notClassified		= 0,	cl_parameter = cl_notClassified,
	cl_other				= 1,
	cl_title				= 2,
	cl_composer				= 3,
	cl_rehearsal			= 4,
	cl_lyric				= 5,
	cl_tempo_marking		= 6,
	cl_dynamic				= 7,
	cl_fingering			= 8,
	cl_pedal				= 9,	cl_ped = cl_pedal,
	cl_fretgrid				= 10,
	cl_chordsymbol			= 11,
	cl_midi					= 12,
	cl_label				= 13,
	cl_articulation			= 14,
	cl_breath				= 15,
	cl_ornament				= 16,
	cl_repeat				= 17,
	cl_copyright			= 18,
	cl_technical			= 19,	cl_bowing = cl_technical,
	cl_pause				= 20,
	cl_mood					= 21,
	cl_tempo_alteration		= 22,
	cl_poet					= 23,
	cl_subtitle				= 24,
	cl_dynamic_alteration	= 25,
	cl_figured_bass			= 26,
	cl_dedication			= 27,	// JA20140331 7.5.230.
	
	cl_TOTAL
};




//								=== TTEXT ===
// Each TTEXT node holds at most MAXTEXT characters.  If a text string is longer
// than this, it is represented by successive TTEXT nodes.
//
// Line-Feeds (end-of-lines) and Carriage-Returns (overwrite current line) are supported
// in TTEXT nodes; however in most cases TPARAM nodes are inserted between TTEXT nodes with
// the required vertical positioning.  See the discussion of TPARAM nodes above.
//
enum{
	MAXTEXT	= (TiliaBytesInMemory-4)	// max number of characters represented in a TTEXT node,
};										// excluding space for the terminating '\0'.

typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is TTEXT
	INT8	Xcontext;			// defines the notation contexts this node applies to

	char	Xtext[MAXTEXT];		// Text character string, up to MAXTEXT characters terminated by zero if shorter;
	//							// n.b. text may continue in a following TTEXT node.
	char	Xtext0;				// Always zero to guarantee to terminate Xtext;  N.B. MUST be same place as in VINFO node;
	//							// (and should be forced to zero if node is read from disk).
	UNINT8	Xtextparent;		// Kind of this TTEXT node's parent (typically TPARAM, default if 0);
	//							// if non-zero, it can be assumed the text is UTF8 encoded.
} TTEXT_node;
#define TTEXT_sizes "11"		// All fields are single bytes.



//								=== LINE ===
// Lines or curves.
//
// During formatting, a PDATA node gets inserted after the LINE node.
// The PDATA node acts as an extension of the LINE node (as well as the
// TPARAM node).
//
typedef struct {
	LINKS					// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;			// kind of node; value is LINE
	INT8	Xcontext;		// defines the notation contexts this node applies to

	XYPOS	Xstartx;		// x offset from notehead to start of line/curve
	XYPOS	Xstarty;		// y offset from notehead to start of line/curve
	XYPOS	Xendx;			// x offset from start of line to end of line
	XYPOS	Xendy;			// y offset from start of line to end of line

	UNBFLD	Xdashlth : 4;	// dashed lines and curves: length of dash segments in
							// dots; zero (NODASH defined below) indicates solid
	UNBFLD	Xgaplth : 4;	// dashed lines and curves: length between gap segments
							// in dots; zero (NODASH defined below) indicates solid

	UNBFLD	Xline_ord : 7;	// if xline_eocvar!=0, xline_ord is ordinal of this variety
							// if xline_eocvar==0, xline_ord nonzero is defined below:

	UNBFLD	Xcurve : 1;		// TRUE for curves, FALSE for lines.  If xcurve is TRUE,
							// then (xstartx, xstarty) give the start of the curve;
							//		(xendx, xendy) give the end of the curve
							//		(xc1x, xc1y) give the first control point
							//		(xc2x, xc2y) give the second control point.
							// If xcurvex is FALSE, then xc1x, xc1y, xc2x, xc2y
							// are corners of a clip rectangle for the line.

	XYPOS	Xc1x, Xc1y;		// offset from start of curve/line to first control point
	XYPOS	Xc2x, Xc2y;		// offset from start of curve/line to second control point

	UNBFLD	Xfattermid : 1; // TRUE if the curve becomes fatter in the middle
	UNBFLD	Xline_fromstaff : 1;	// OBSOLETE, replaced by Xline_rely
							// like xfromstaff for TPARAM: was TRUE if
							// xstarty was the offset from the center staff line;
							// FALSE if xstarty was the offset from the notehead y.
	UNBFLD	Xline_eachsys : 1;
							// TRUE if this was inserted using Duplicate At Each System.
							// This has no special meaning; it is only used by the editor
							// if the user uses Clear At Each System. This field is
							// provided to allow the user to undo an accident.
	UNBFLD	Xline_eachpage : 1;
							// TRUE if this was inserted using Duplicate On Each Page.
							// This has no special meaning; it is only used by the editor
							// if the user uses Clear On Each Page. This field is
							// provided to allow the user to undo an accident.
	UNBFLD	Xline_everypart : 1;
							// TRUE if this was inserted using Duplicate In Every Part.
							// If this bit is set, the annotation will only print if
							// this is the top staff in the current notation context; it is
							// also used by the editor if the user uses Clear In Every Part.
	UNBFLD Xline_drawmode:2;// like xdrawmode for LINE: ORMODE or BICMODE
	UNBFLD Xline_pagemargin : 1;	// OBSOLETE, replaced by Xline_relx
							// like xpagemargin for TPARAM: was TRUE if the line
							// should have been placed relative to the page margins,
							// FALSE if xstartx was relative to notehead x.

	UNINT8 Xlinewidth;		// width of line in tenth point units
	UNBFLD Xmetaline    :4;	// nonzero if this LINE node represents a hairpin, tuplet
							// bracket, a bracket over notes, or a rectangle.
	UNBFLD Xline_color  :4; // color index into color table

	UNBFLD Xline_eocvar;	// zero if this is a regular annotation; else it is the variety
							// of graphic being modified by this LINE node.  This is for
							// Extended Option-click.

	UNBFLD Xline_number :4;	// Number associated with Line; beginning and end of extendable lines must carry same number
	//						// Should be non-zero if possible.  A zero value indicates potential conflict.
	UNBFLD Xline_hidden :1;	// Line is hidden
	UNBFLD Xline_extend :1;	// temporary flag during editing, indicating extendable  
	UNBFLD Xline_resize :2;	// Reserved field, will replace the use of Xline_ord for flagging components of resizable lines
	
	UNBFLD Xline_rely	:4;	// what y positioning is relative to (Y anchor)
	UNBFLD Xline_relx	:4;	// what x positioning is relative to (X anchor)
	
	TiliaRGBA	Xline_rgba;	// Reserved for generalized colouring
	// 46 bytes up to here
} LINE_node;
#define LINE_sizes "11xxxx(44)(71)xxxx(1111121)1(44)1(4112)(44)1111"

enum{
	REGLINE				= 10,	// regular line width
	MEDLINE				= 8,	// Value used internally
	THINNERLINE			= 5,
	THINLINE			= 2,	// default used for stems, staff lines, etc
	FATLINE				= 25,	// default used in beaming notes
	GRACE_FATLINE		= 15,	// default used in beaming grace notes
	GROUP_SELECT_LW		= 5,	// default line width used for sample_line in GROUP_SELECT
	//							// must be different from REGLINE, THINLINE, and FATLINE so that
	//							// none of the width buttons in the annotation controls are selected 
};
enum{
	MAX_DASHLTH			= 15,		// maximum value for xdashlth
	MAX_GAPLTH			= 15,		// maximum value for xgaplth
	NODASH				= 0			// indicates solid line; in subroutine calls,
	// used for both xgaplth and xdashlth
};
enum {
	STRAIGHT_LINE		= 0,
	HAIRPIN				= 1,	// xmetaline: a hairpin (one line of a hairpin)
	TUPLET_BRACKET		= 2,	// xmetaline: a tuplet bracket
	PLAIN_BRACKET		= 3,	// xmetaline: a plain bracket
	RIGHT_BRACKET		= 4,	// xmetaline: a right-only bracket (e.g, 8va line)
	LEFT_BRACKET		= 5,	// xmetaline: a left-only  bracket
	RECTANGLE			= 6,	// xmetaline: a full rectangle
	RIGHT_RECTANGLE		= 7,	// xmetaline: a right-only  rectangle
	LEFT_RECTANGLE		= 8,	// xmetaline: a left-only   rectangle
	BOTTOM_RECTANGLE	= 9,	// xmetaline: a bottom-only rectangle
	TOP_RECTANGLE		= 10,	// xmetaline: a top-only    rectangle
	tot_METALINE
};
// Resizing lines/hairpins/8va, etc have two LINE nodes; both LINE nodes have xline_eocvar==0.
// The LINE node anchoring the beginning of the line has xline_ord==BEGLINE.
// The LINE node anchoring the end of the line has xline_ord==ENDLINE.
// In the next file version, a separate field, xline_resize, will be used.
// The two LINE nodes can have different xstartx and xendx fields, but must have matching:
//	-	xstarty and xline_rely
//	-	xendy
//	-	xcurve and xmetaline
//	-	xline_number	// this is used to try to ensure uniqueness, should be non-zero if possible
//
// A LINE node with xline_ord==MIDLINE (and xline_eocvar==0) is a Lime-generated node,
// without any permanent information.  These are created and destroyed each time Lime
// reformats the page; they aid in drawing resizing lines that span system breaks.
//
enum{
	BEGLINE				= 3,	// xline_ord value for LINE node anchoring the start-of-line
	MIDLINE				= 2,	// xline_ord value for temporary LINE node in middle of multi-system line
	ENDLINE				= 1,	// xline_ord value for LINE node anchoring the end-of-line
	ALLLINE				= 0		// xline_ord value for LINE node that is not resizable (0)
};
// all these tests include check for node type
//
#if CURRENT_FILE_FORMAT_NUMBER <= GENERAL_RELATIVITY_VERSION
# define is_RESIZE(p)	(is_xkind(p,LINE) && !xline_eocvar(p) ?  xline_ord(p) : 0)
# define is_ENDLINE(p)	(is_xkind(p,LINE) && !xline_eocvar(p) && xline_ord(p) == ENDLINE)
# define is_MIDLINE(p)	(is_xkind(p,LINE) && !xline_eocvar(p) && xline_ord(p) == MIDLINE)
# define is_BEGLINE(p)	(is_xkind(p,LINE) && !xline_eocvar(p) && xline_ord(p) == BEGLINE)

# define set_ENDLINE(p)	(xline_eocvar(p) = 0, xline_ord(p) = xline_resize(p) = ENDLINE)
# define set_MIDLINE(p)	(xline_eocvar(p) = 0, xline_ord(p) = xline_resize(p) = MIDLINE)
# define set_BEGLINE(p)	(xline_eocvar(p) = 0, xline_ord(p) = xline_resize(p) = BEGLINE)
# define set_ALLLINE(p)	(xline_eocvar(p) = 0, xline_ord(p) = xline_resize(p) = ALLLINE)
#else
# define is_RESIZE(p)	(is_xkind(p,LINE) ?  xline_resize(p) : 0)
# define is_ENDLINE(p)	(is_xkind(p,LINE) && xline_resize(p) == ENDLINE)
# define is_MIDLINE(p)	(is_xkind(p,LINE) && xline_resize(p) == MIDLINE)
# define is_BEGLINE(p)	(is_xkind(p,LINE) && xline_resize(p) == BEGLINE)

# define set_ENDLINE(p)	(xline_resize(p) = ENDLINE)
# define set_MIDLINE(p)	(xline_resize(p) = MIDLINE)
# define set_BEGLINE(p)	(xline_resize(p) = BEGLINE)
# define set_ALLLINE(p)	(xline_resize(p) = ALLLINE)
#endif




//							   === xxxFNAME ===
// Obsolete node to specify a font number and the associated font name.
// This definition is kept only for converting old Lime pieces.
//
enum{ xxxMAXFNAME = 16 };		// obsolete max number of characters in xxxFNAME node

typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is xxxFNAME
	INT8	Xcontext;			// defines the notation contexts this node applies to

	INT16	XXXfontnum;			// obsolete: font number, as used in TPARAM nodes
	char	XXXfontnam[xxxMAXFNAME];// obsolete: name for the font, must be terminated with '\0'
} xxxFNAME_node;
#define xxxFNAME_sizes "1121111111111111111"  // for obsolete node





//							   === NIFFX ===
//
// NIFFX nodes are created when importing NIFF files.  They store x-location
// from the NIFF representation.  They are written to disk, but should be deleted
// once all corrections have been made.
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is NIFFX
	INT8	Xcontext;			// defines the notation contexts this node applies to

	XYPOS	Xniffx;				// NIFF x location for the note's timeslice; this is in Tilia units
								// value is after offset/skew correction for rotated scan image.
	UNBFLD	Xniffrhythm : 1;	// obsolete: TRUE to mark probable rhythm errors
	UNBFLD	notused : 7;
	UNINT8	notusedyet;
	XYPOS	Xniffy;				// NIFF staff y location; this uses Tilia units except that origin is at top;
								// value is after offset/skew correction for rotated scan image.
} NIFFX_node;
#define NIFFX_sizes "11x(17)1x"


//								=== BAR ===
//
typedef struct{
	INT16		page;			// temporary:	page in which this BAR occurs
	UNINT16		bpm;			// temporary:	prevailing tempo bpm
	UNINT8		lgval : 4;		// temporary:	prevailing tempo lgval
	UNINT8		dots  : 4;		// temporary:	prevailing tempo dots count
	UNINT8		hearcount;		// temporary:	count how often Hear option played through this barline
	INT8		printing;		// temporary:	prevailing voice printing staff + 1
	INT8		tsigparity;		// temporary:	parity  of BAR count since TSIG node
	DefTiliaPtr tsigptr;		// temporary:	pointer to prevailing TSIG node, if any
} BAR_node_heartemp;

typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is BAR
	INT8	Xcontext;			// defines the notation contexts this node applies to

	INT8	Xdouble;			// type of bar line
	INT8	Xbarbetween;		// temporary data while Lime runs, from BARSEP node:
	//							//	>0 for bar between staff and staff+1; ==0 for bar just on staff;
	//							//	<0 for totally invisible BAR.
	INT8	Xleftdots;			// TRUE if repeat dots on left of bar
	INT8	Xrightdots;			// TRUE if repeat dots on right of bar
	INT8	Xunnumbered;		// TRUE if measure is not counted in the measure numbering

	UNBFLD	Xwings : 1;			// TRUE to use thicker repeat signs with wings
	UNBFLD	Xrestart : 1;		// TRUE if measure numbering restarts at xrestart_num
	UNBFLD	Xhiden : 1;			// TRUE avoids printing measure number at start of sys
	UNBFLD	Xchant : 1;			// TRUE for chant bar line
	UNBFLD	Xbarpickup  : 2;	// TRUE if this precedes a pickup measure (used for hear metronome)
	UNBFLD	Xbarwithprev: 1;	// TRUE to always combine with previous BAR when Lime Lighter scrolling
	UNBFLD	notusedyet1 : 1;

	XYPOS	Xbarx;				// temporary data while Lime runs: bar x location
	XYPOS	Xbarorigin;			// temporary data while Lime runs: bar relocation x
	INT8	Xnotstaff;			// TRUE means: don't draw bar on the staff itself
								// (from top to bottom staff line assuming a five-line
								// staff).  Together with BARSEP nodes, this
								// controls where the bar line is drawn.
	INT8	Xdashed;			// TRUE if the bar line is dashed
	XYPOS	Xscrbarx;			// temporary data while Lime runs: screen x location
	UNINT16	Xrestart_num;		// if xrestart is TRUE, then restart bar numbering with this value.
	INT8	Xgoto_type;			// for hear option: see values defined below
	INT8	notusedyet;	
	UNINT16	Xgoto_bar;			// for hear option: continue at this measure; used if
								// Xbar_hear.hearcount and xgoto_type have certain values
	UNINT16	notusedyet2;
	
	INT8	notusedyet8[8];
	
// 38 bytes to here
// The following are temporary data  while Lime runs, used in formatting and playback (mostly play-back).
// It is possible that, in the future, if a BAR node gets overloaded, the hearing information
// will be moved to a new BARINFO node; however while there is space it is more efficient to keep
// the data here.  These data have no meaning whatsoever on disk and are ignored when pieces are read.
//
	INT8	Xbarused2format;	// temporary:	identifies that this BAR was used for formatting
	INT8	Xbar_goneto;		// temporary:	identifies a bar that is continued to (reserved)
	BAR_node_heartemp Xbar_hear;// temporary:	flags, etc for the Hear Option (12 bytes)

} BAR_node;
#define BAR_sizes "1111111(1111211)xx11x2112211111111"	// n.b. does not include truly temporary data
//
enum{
	INVISIBLEBAR	= -128,		// bar line not drawn at all (currently any negative value)
	REGBAR			= 0,		// single bar
	DOUBLEBAR		= 1,		// double bar
	ENDBAR			= 2,		// end of piece double bar: second bar fatter
	BEGBAR			= 3,		// first bar fatter
	TRIPLEBAR		= 4,		// thin fat thin (forced if winged repeat)
	
	NO_GOTO			= 0,		// xgoto_type value: continue playing regularly
	FIRST_ENDING	= 1,		// xgoto_type value: start of first ending
	SECOND_ENDING	= 2,		// xgoto_type value: start of second ending
	FIRST_TIME		= 3,		// xgoto_type value: first time go to xgoto_meas
	SECOND_TIME		= 4,		// xgoto_type value: second time go to xgoto_meas
	THIRD_TIME		= 5,		// xgoto_type value: third time go to xgoto_meas
	//							// etc.  -ve  means: -nth time or greater go to xgoto_meas
	
	XBARX_NOT_SET	= UNDEFXY	// initial (undefined) value for Xbarx and Xscrbarx	
};



//								  === BARSEP ===
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is BARSEP
	INT8	Xcontext;			// defines the notation contexts this node applies to

	// This node exists before the same note as a BAR node, if the  bar is to be separated
	// (not drawn between staff, staff+1).  This is the complement of the BAR node's old xbetween()
	// field; this function now has its own node, so that it can depend on notation contexts.

	INT8	Xbarsbetween;		// Always FALSE currently, meaning no bar line drawn between staves.
								// TRUE to draw line between staves normally.
	INT8	Xbarsreserve;		// (reserved)
} BARSEP_node;
#define BARSEP_sizes "1111"

//								   === ENDSYST ===
//
enum{
	BRACKET_BETWEEN		= 0,			// bracket on staff and to next staff
	BRACKET_ON_STAFF	= 1,			// bracket is drawn on staff
	NO_BRACKET			= 2,			// no bracket drawn for this staff
	
	NO_BRACE			= 0,			// no curly brace drawn for this staff
	BRACE_BETWEEN		= 1,			// brace between this staff and next staff
	
	NO_SYS_TIE			= 0,			// indicates no tie for xsys_tie fields
	DASHED_SYS_TIE_BIT	= 0x80,			// bit in sys_tieup[] and sys_tiedown[] for dashed
	MOD12_SYS_TIE_FIELD	= 0x7f,			// field in sys_typeup[] and sys_tiedowm[] for mod12
	
	MAX_SYS_TIE			= 5				// Maximum number of ties over the end of the system
	//									// for one voice; there are this many bowed up and
	//									// again this many bowed down.
};
enum{
	ENDSYST_maskEndpage			= 0x3F,	// on disk top 2 bits of Xendpage must always be 0
	ENDSYST_disabledEndpage		= -1,	// Disabled endpage, but active ENDSYST
	ENDSYST_disableEndpage_bit	= 0x80,	// bit to disable Xendpage
	ENDSYST_invisibleEndpage	= ENDSYST_disableEndpage_bit | 1
};
// With the obvious exception of the end page, end piece and start piece flags, and 
// the bracket and brace drawing, and the hangover (and the unused Xttstart),
// most of the ENDSYST fields are set and cleared by the formatter.

//								   == ENDSYST structure ==
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is ENDSYST
	INT8	Xcontext;			// defines the notation contexts this node applies to

	INT8	Xendpage;			// TRUE if it is end of page; -ve used internally for ticker-tape  to
								// flag temporarily disabled.	= -1	treat as normal ENDSYST
								//								< -1	concatenated ENDSYST
	INT8	Xendpiece;			// TRUE if it is end of piece;
	INT8	Xstartpiece;		// TRUE if it is start of piece

	UNBFLD	bunused		: 1;	// this bit is not used
	UNBFLD	Xsys_gliss	: 1;	// TRUE is there is a glissando off the end of the system;
								// this field is set by the formatter, so that this
								// information is available for the next system/page.
	UNBFLD	Xsys_beamup : 1;	// TRUE if there is a up-stem beam off end of system;
								// this field is set by the formatter, so that this
								// information is available for the next system/page.
	UNBFLD	Xsys_beamdown : 1;	// TRUE if there is a down-stem beam off end of system;
								// this field is set by the formatter, so that this
								// information is available for the next system/page.
	UNBFLD	Xbracket : 2;		// controls bracket drawing at start of system.  If xstartpiece
								// is set on this ENDSYST node, this field is ignored and
								// the xfirst_bracket field in the first PRINT node is used
								// instead; this is because the first ENDSYST node is shared
								// among all notation contexts, but bracket controls should not be.
								// The following are values for xbracket and xfirst_bracket:
	UNBFLD	Xbrace : 2;			// controls curly brace drawing at start of system. If xstartpiece
								// is set on this ENDSYST node, this field is ignored and
								// the xfirst_brace field in the first PRINT node is used
								// instead; this is because the first ENDSYST node is shared
								// among all notation contexts, but bracket controls should not be.
								// The following are values for xbrace and xfirst_brace:

	INT32	Xttstart;			// This field is used only if xstartpiece==TRUE.
								// It indicates the time that should elapse
								// before the voice starts playing (analogous to
								// the xttn field in NOTE nodes).  This allows
								// one voice to start with grace notes while
								// another voice has an implied rest. [reserved]

	INT8	Xsys_tieup[MAX_SYS_TIE];
								// xmod12 values for up-ties off end of system;
								// a NO_SYS_TIE value indicates no tie.  These fields
								// are set and used by the formatter.
	INT8	Xsys_tiedown[MAX_SYS_TIE];
								// xmod12 values for down-ties off end of system;
								// a NO_SYS_TIE value indicates no tie.  These fields
								// are set and used by the formatter.
	
	INT8	Xsys_voice;			// voice number which this ENDSYST node belongs to; this
								// is set by findpage() in page.c, and it is used in
								// param.c when looking for PARAM nodes before this page.
	INT8	unused;
	
	// These slur fields are set (and cleared) by the formatter, so that this
	// information is available for the next system/page.  Apart from the
	// indicators that there is a slur, the staff, layer and grace values
	// provide an index into the slur information buffer that was in use 
	// in this system.
	//
	UNINT8	Xsys_staff_slurup;		// Staff number of  up-slur off end of system
	UNBFLD	Xsys_layer_slurup	:4;	// Layer			  ""
	UNBFLD	Xsys_grace_slurup	:3;	// Grace slur type	  ""
	UNBFLD	Xsys_slurup			:1;	// TRUE if there is a up-slur off end of system
	
	UNINT8	Xsys_staff_slurdown;	// Staff number of  down-slur off end of system
	UNBFLD	Xsys_layer_slurdown	:4;	// Layer			  ""
	UNBFLD	Xsys_grace_slurdown	:3;	// Grace slur type	  ""
	UNBFLD	Xsys_slurdown		:1;	// TRUE if there is a down-slur off end of system	
	
	// The following field identifies the amount of time the last chord before the ENDSYST
	// hangs over to the next system.  In normal music this will be zero, but can be non-zero
	// if a system break occurs at a place that is not a timing boundary for this voice..  
	//
	INT32	Xsys_hangover;			// Time of last note to hang over to next system
	
	// 30 bytes to here
	
	INT8	unused8[8];				// Free
	
	// This field is set by the formatter, so that the data are available for the next system/page.
	// It provides the duration so far of a tuplet group that extends off the end of system.
	// A non-zero value indicates that it comes on to the next system.
	// Because only one tuplet is shown automatically, there is only a need for one tuple field.
	//
	INT32	Xsys_tupledur;			// Duration so far in a tuplet off end of system

	// 42 bytes to here
} ENDSYST_node;
#define ENDSYST_sizes "11111(111122)41111111111111(431)1(431)4111111114"




//								=== PRINT ===
// Tilia PRINT nodes are generated by the "Parts and Voices..." menu item in Lime.
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is PRINT
	INT8	Xcontext;			// defines the notation contexts this node applies to

	INT16	Xsstaff;			// staff number in system; NOSTAFF indicates Don't Print (if
								// xlist!=NOLIST) or Deactivated Voice (if xlist==NOLIST).
	INT16	Xprintdir;			// UP, DOWN or UPORDOWN for print direction
								// of the voice on the staff

	INT16	Xslines;			// number of staff lines (usually 5)
	
	INT8	Xprinthide;			// hook - to print but hide.  Set to UNDEF08 by MusicXML import
								// when rests are hidden for hopping or annotation reasons.

	INT8	Xlhrh;				// temporary while Lime runs: indicates a voice switching
								// between staves, potentially beaming between the staves

	INT8	Xxxforcues;			// OUTDATED: TRUE if staff is cue-size staff

	UNBFLD	Xfirst_bracket : 2; // controls bracket drawing at start of system for the first
								// system in the piece; if this is not the first PRINT node on
								// the voice, this field is ignored.  See comments on xbracket
								// field definition for the ENDSYST node.
	UNBFLD	Xfirst_brace : 2;	// controls brace drawing at start of system for the first
								// system in the piece; if this is not the first PRINT node on
								// the voice, this field is ignored.  See comments on xbrace
								// field definition for the ENDSYST node.
	UNBFLD	Xtablature : 2;		// nonzero if tablature notation is to be used;
								// normally the user also selects 6 staff lines.
								// 1 means stemless tablature, 2 is tab with stems,
								// 3 is piano tablature
	UNBFLD	Xnoledgers : 2;		// Non zero if no ledger lines,
	//							// UPORDOWN(3) = none at all; DOWN(1) = none below; UP(2) = none above.
	INT16	Xstaffsize;			// points between staff lines (times STD_SF)

	UNBFLD	Xpartstaff : 4;		// staff number within part (if xsstaff != NOSTAFF), 0 is first staff in part
	UNBFLD	Xpartstaves : 4;	// number of staves for part (0 means xpartstaff not valid)
	INT8	Xlist;				// for Lime internal use: list entry in Voices... dialog;
								// value is 1..N, NOLIST means "deactivated voice".
	INT8  Xlayer;				// nonzero: layered voice (print using separate internal staff representation);
								// special value ALTLYR automatically set by altlayer().

	INT8  unused;
} PRINT_node;
#define PRINT_sizes "11222111(2222)2(44)11"

enum{
	PIANOTAB_RH_SLINES	= 7,	// number of staff lines in piano tablature right hand
	PIANOTAB_LH_SLINES	= 8,	// number of staff lines in piano tablature left hand
	MAX_SLINES			= 12	// max number of staff lines in non-tablature staff
};
enum{
	LH					= 1,	// xlhrh value: switch to lower staff
	RH					= 2,	// xlhrh value: switch to higher staff
	NOLHRH				= 0		// xlhrh value: no switch
};
enum{
	TAB_STEM			= 1,	// xtablature for guitar tablature with stems
	TAB_NOSTEM			= 2,	// xtablature for stemless guitar tablature
	TAB_PIANO			= 3,	// xtablature for piano tablature
	NOT_TAB				= 0		// xtablature for normal notation
};




//								=== SDRAG ===
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is SDRAG
	INT8	Xcontext;			// defines the notation contexts this node applies to

	XYPOS	Xstaffdrag;			// Displace the current staff from its normal
								// y position by this amount.  This field is set
								// when an individual staff is draged.
	INT16	Xsdrag_sstaff;		// staff number within system for which staff drag was made for;
								// avoids staff jumping if voices switch between staves
} SDRAG_node;
#define SDRAG_sizes "11x2"



//								=== BEAMING ===
//
enum{
	MAXBVALUE	= 4				// maximum number of beaming values per voice
};
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is BEAMING
	INT8	Xcontext;			// defines the notation contexts this node applies to

	INT32	Xsbvalue;			// Subbeaming value (16 bit whole-note, 16 bit fractional).
								// This value is used to determine sub-beaming boundaries.
	INT32	Xbvalue[MAXBVALUE]; // Beaming value (16 bit whole-note, 16 bit fractional).
								// A list of beat values is given for each voice.
								// If less than MAXBVALUE values are used, the list of values
								// is terminated by a zero value.
								// The list is repeated if necessary for long measures.
								// These values are used to determine beaming boundaries.
	INT32	unused32;
	INT8	Xbeams_endsyst;		// TRUE  to terminate beams at all system ends;
	//							// FALSE means they are only teminated if there is an associated BAR.

	INT8	unused;
} BEAMING_node;
#define BEAMING_sizes "1144444411"


//								=== SPACE ===
//
// This node is supported only for backward compatability; xleavespace annotations
// and chord drag make this an unnecessary node type.
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is SPACE
	INT8	Xcontext;			// defines the notation contexts this node applies to

	INT16	Xspace;				// Make sure there is at least this much blank space
								// before the note.  Supported for backward compatability;
								// replaced by xleavespace and chord drag.
} SPACE_node;
#define SPACE_sizes "112"



//								=== CMOVE ===
// This is used to move one chord to a new x location.  It does not affect any
// other chords.
//
// It can also be used to move rests to a new y location.
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is CMOVE
	INT8	Xcontext;			// defines the notation contexts this node applies to

	XYPOS	Xchordmove;			// Displace the chord from its normal x position
								// this amount; do not relocate any other chords.
								// This field is set after all objects have been
								// placed, and an individual chord is moved.  This
								// can be used to skew chord placement on different
								// staves: chords at the same time may not plot
								// at the same x location.
} CMOVE_node;
#define CMOVE_sizes "11x"



//								=== AMOVE ===
// This is used to move one note's accidentals to a new x location.
// It does not affect any other notes or their accidentals.
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is AMOVE
	INT8	Xcontext;			// defines the notation contexts this node applies to

	XYPOS	Xaccmove;			// Displace the note's accidental from its normal
								// x position this amount.
} AMOVE_node;
#define AMOVE_sizes "11x"






//								=== RMOVE ===
// This is used to move a rest to a new y location.
// It does not affect any other notes or rests.
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is RMOVE
	INT8	Xcontext;			// defines the notation contexts this node applies to

	XYPOS	Xrestmove;			// Displace the rest from its normal y position by
								// this number of staff spaces.
} RMOVE_node;
#define RMOVE_sizes "11x"

enum{
	MAX_RESTMOVE	= 20		// move no more than 20 staff spaces up or down
};



//								=== ZONE ===
// ZONE nodes bound relocation zones for notes.  They specify an x location (as a
// permil of system width) at which the associated chord should be located; the
// chord is moved to this location, and the chords before and after this chord are
// relocated appropriately.  This allows the user control over general note density
// in different parts of the system; it also allows the user to line up measure
// bars on different systems.
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is CMOVE
	INT8	Xcontext;			// defines the notation contexts this node applies to

	INT16	Xzone_xpermil;		// Displace the chord from its normal x position
								// to this position; relocate chords before and after
								// this chord appropriately. This is used to make
								// bar lines on different systems line up, or to make
								// a measure wider or narrowed than the default.
								// This quantity is given in permil (tenth percents)
								// of the total system width.
	INT32	Xzone_wholec;		// Whole note count at start of zone.
	INT32	Xzone_sys_wholec;	// Whole note length of whole system; this is kept to
								// help detect ZONE nodes which are out of date due to
								// systems being split or combined, or due to measures
								// being deleted, etc. If the length of the system does
								// not match xzone_sys_wholec, then the ZONE node is out
								// of date and will be removed.
	INT8	Xzone_bar;			// TRUE if we are specifying the previous bar location
	INT8	unused;
} ZONE_node;
#define ZONE_sizes "1124411"


//								=== PARAM ===
// The "Parameters" menu option in Lime creates hidden text annotations, which specify
// changes to various music formatting parameters.  When Lime formats a page of music for
// display,
//		(1) it deletes all existing PARAM nodes,
//		(2) it interprets hidden text annotations and inserts corresponding PARAM nodes,
//		(3) it uses these PARAM nodes (for efficient look-up) in later formatting passes.
// PARAM nodes are recreated each time the page is formatted; thus if the user deletes, moves,
// or modifies the corresponding hdden text annotation, the PARAM nodes will reflect the change
// the next time the page is reformatted.  PARAM nodes are stored on disk in the Tilia file.
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is PARAM
	INT8	Xcontext;			// defines the notation contexts this node applies to

	DefTiliaPtr Xprev_param;	// temporary pointer while Lime runs: points to the previous
								// PARAM node on this voice.  This reverse linked list
								// is used to find PARAM nodes on previous pages quickly.
	INT16	Xparam_index;		// The parameter number which is being modified by
								// the param node; as described above, use
								// Option-Parameters in Lime to see these numbers

	UNBFLD	Xparam_small : 1;	// TRUE if the small-size (cue or grace note size) version
								// of the parameter is to be modified; FALSE if the normal
								// size version of the parameter is to be modified.
	UNBFLD	unusedbits : 7;		// bits available for future use

	INT8	unusedbyte;			// byte available for future use
	PARAM_VALUE Xparam_value;	// Either a specification for a new character mapping,
								// an integer value, or a floating value: depends on type
								// of parameter indicated by Xparam_index.

	INT16	Xparam_stsize;		// Size of staff at time of parameter specification.
} PARAM_node;
#define PARAM_sizes "1142(17)1+2"	/* + indicates PARAM_VALUE union, 10 bytes */

enum{							// Bytes offset between Xparam_value and Xparam_index;
	PVAL_TO_PINDEX = 4			// kludge for identifying index in endedness conversion.
};



//								=== SPACC ===
//
// If a note has one or more special accidentals, these are stored in the
// SPACC node.
//

enum{ MAXSPACC	= 15 };			// max number of accidental symbols before a note

typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is SPACC
	INT8	Xcontext;			// defines the notation contexts this node applies to

	INT8	Xnumspacc;			// number of accidentals before note; values range
								// from 1 to MAXSPACC
	INT8	Xspacc[MAXSPACC];	// list of accidentals for note; value between
								// DOUBLEFLAT and DOUBLESHARP indicates standard symbols,
								// value of SPECIALACC+N-1 is special accidental
								// symbol N in Parameters.
} SPACC_node;
#define SPACC_sizes "111111111111111111"

enum{
	SPECIALACC	= 6				// xspacc[]>SPECIALACC means special accidental symbol
};



//								=== NEST ===
//
// If a note is a nested tuplet, the nested outer tuplet specification is stored in the
// NEST node.
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is NEST
	INT8	Xcontext;			// notation context, should always be 0, all contexts.

	UNINT8	Xnest_tupl;			// 3 for nested triplets  see xnest_intimeof field below
	UNINT8	Xnest_intimeof;		// used with xnest_tupl field: "xnest_tupl in the time of xnest_intimeof";
								// examples: "3 in the time of 2" or "5 in the time of 7";
								// this field is shifted left by 2 to allow things like
								// "5 quarters in the time of 7 eighths" expressed by
								// "5 in the time of 3.5"
} NEST_node;
#define NEST_sizes "1111"


//								=== COLOR ===
//
// If a COLOR node precedes a NOTE, the noteheads (or rest) will be displayed
// in the specified color.
//

typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8		Xkind;			// kind of node; value is COLOR
	INT8		Xcontext;		// defines the notation contexts this node applies to

	RGBColor	Xxxcolor;		// OUTDATED color: red, green, blue  2 bytes each, 6 total
	INT8		Xnote_color;	// index into color table; standard and alternate coloration
	UNINT8		Xcolor_kind;	// Node type being coloured, default type of next node (typically a NOTE)
	TiliaRGBA	Xcolor_rgba;	// Reserved for generalized colouring
} COLOR_node;
#define COLOR_sizes "11222111111"




//								=== STMOD ===
//
// If a STMOD node precedes a NOTE, the note's slur or tie will be modified by the
// STMOD specifications.  These allow special ties and slurs, such as those required
// for marking a slurred-to note at the beginning of a second ending.  This also
// provides for a second slur, as needed for a trill that is under a long slur and also
// requires a slur to nachschlag notes (after-the-note grace notes).
//

typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is SLURMOD
	INT8	Xcontext;			// defines the notation contexts this node applies to

	UNBFLD	XhangRslur : 1;		// break slur drawn to right
	UNBFLD	XhangLslur : 1;		// break slur drawn to left
	UNBFLD	Xleftonlyslur : 1;	// no slur to right at all (hangRslur ignored for this)

	UNBFLD	XhangRtie : 1;		// break tie drawn to right
	UNBFLD	XhangLtie : 1;		// break tie drawn to left
	UNBFLD	Xleftonlytie : 1;	// no tie to right at all (hangRtie ignored for this)

	UNBFLD	Xsplitslur : 1;		// slur split at note (slur ends, another starts at notehead)
	UNBFLD	Xaltslursize : 1;	// xslur field refers to slur of type xgrace^xaltslursize


	// Info for additional slur.  The additional slur is only present if xa_slur != NONE.
	// The xa_slur is of the opposite size (grace-type) of the xslur.
	UNBFLD	Xa_slur : 2;		// xslur for additional slur
	UNBFLD	Xa_crawlslur : 1;	// xcrawlslur for additional slur
	UNBFLD	Xa_hangRslur : 1;	// xhangRslur for additional slur
	UNBFLD	Xa_hangLslur : 1;	// xhangLslur for additional slur
	UNBFLD	Xa_leftonlyslur:1;	// xleftonlyslur for additional slur
	UNBFLD	Xa_splitslur : 1;	// xsplitslur for additional slur
	UNBFLD	Xa_dashslur : 1;	// xdashslur for additional slur

	UNBFLD	Xa_bendslur : 1;	// xbendslur for additional slur

	// Staccato modifier.
	UNBFLD	Xstacissimo : 1;	// if there is a staccato dot, it is staccatissimo
	
	// Slur attachment preference
	// Values as defined for options (SlursPrefer_ etc)
	//
	UNBFLD	Xslurattachment	: 3;
	UNBFLD	Xa_slurattachment:3;
	
} TSMOD_node;
#define TSMOD_sizes "11(11111111)(2111111)(1133)"



// ////////////// Nodes that are mostly internal but can be written when saving graphics //////////////
//				  ----------------------------------------------------------------------

//								=== NTDATA ===
//
// The NTDATA node is a temporary node used in formatting music for
// printing.  It always immediately follows a NOTE node.  A pointer through
// the music binary always points at the NOTE node; elements in the NTDATA
// node are accessed through the 'xnext' pointer of the NOTE node.
// To the programmer it is transparent whether he is accessing a field
// from the NOTE node or the NTDATA node.
//
// For the last note in a staff-list chord, the NTDATA node is immediately followed
// by CDDATA,  CDDATA2, etc.
//
enum{							// Xaccx values
	MAX_ACCMOVE		= BIG16,	// 
	ACC_NOT_PLACED	= UNDEF16	// Xaccx value indicating accidental not placed;	
};
enum{							// Xtabstr values
	MAX_STRING		= 7,		// at most 7-string guitar tablature
	TAB_REST		= 8			// fake "string" for guitar tab rests
};
enum{							// Xrestchar values
	RCMEAS			= 2,		// rest prints as a bar (several measures are combined)
	RCWHOLE			= 1,		// rest prints as a whole rest
	RCNORMAL		= 0			// rest is not combined
};
#define isRCCOMBINED(rc) ((rc)!=RCNORMAL)	// anything but zero is combined; used as an initializer in xhdy.c and placecd.c

//								== NTDATA structure ==
typedef struct {
		LINKS					// macro to make next and back pointers for Lime
		UNINT8	Xkind;			// kind of node value is CDDATA
		INT8	Xcontext;		// notation context, should always be 0, all contexts,
		//						// except in the rare case of a very temporary note, while Lime runs.
#if 0	// Would be more efficient not to use bit fields
		// Check GoodFeel doesn't use these fields.
		//
		UNINT8	Xforced	 ;		// TRUE if stem direction forced UP or DOWN
		UNINT8	Xendbeam ;		// TRUE if this is end of note-stem beam
		UNINT8	Xtmpbeam ;		// TRUE if this is temporary end of beam
		UNINT8	Xcluster ;		// TRUE if noteheads on both sides of stem
		UNINT8	Xendscd	 ;		// TRUE if end of staff chord
		UNINT8	Xstmdir	 ;		// printing stem direction: UP or DOWN
		UNINT8	Xtmpdir	 ;		// TRUE if this is temporary stem direction; used for grace notes
		
		UNINT8	Xtabtyp	 ;		// tablature field value of SDATA for staff
		UNINT8	Xtabstr	 ;		// string number for guitar tablature
		
		UNINT8	Xtieup	 ;		// TRUE if tie is drawn with upward curvature (if there is a tie)
#else
		UNBFLD	Xforced	 : 1;	// TRUE if stem direction forced UP or DOWN
		UNBFLD	Xendbeam : 1;	// TRUE if this is end of note-stem beam
		UNBFLD	Xtmpbeam : 1;	// TRUE if this is temporary end of beam
		UNBFLD	Xcluster : 1;	// TRUE if noteheads on both sides of stem
		UNBFLD	Xendscd	 : 1;	// TRUE if end of staff chord
		UNBFLD	Xstmdir	 : 2;	// printing stem direction: UP or DOWN
		UNBFLD	Xtmpdir	 : 1;	// TRUE if this is temporary stem direction;
								// used for grace notes

		UNBFLD	Xtabtyp	 : 3;	// tablature field value of SDATA for staff
		UNBFLD	Xtabstr	 : 4;	// string number for guitar tablature

		UNBFLD	Xtieup	 : 1;	// TRUE if tie is drawn with upward curvature (if there is a tie)

		UNINT8  free8[8];		// spare bytes
#endif
	union{struct{
		XYPOS	Xhdx;			// x location of notehead
		XYPOS	Xhdy;			// y location of notehead
	  };
		DefTiliaPtr Xnote_pseudo;// Internal use ONLY! 
		//						//  Used as temporary pointer to a real note in a chord
		//						//  if this note is not seen.  The Xhdx and Xhdy fields
		//						//  will then be set to something sensible
	};
		UNINT8	Xstsize;		// staff size

		UNBFLD	Xrestchar : 2;	// rest character used for combined rests: RCNORMAL, etc.

		UNBFLD	Xscrrestchar : 2;
								// value of "xrestchar" in offscreen image.
	
		UNBFLD	sparebit1:1;
	
		UNBFLD	Xtup_info  : 1; // tuplet information: TRUE means this is part of a tuplet
								// group and the xtup_start, xtup_end, xtup_mark fields
								// in the CDDATA node have more information.  FALSE means
								// there is no special tuplet information for this note
								// (the note may or may not be part of a tuplet group).
								// This will only be set TRUE on a note which is the last
								// in its stem direction.  See tupgroups.c.
		UNBFLD	Xother_staff : 1;
								// the stem goes between staves, and this note is to go on
								// the "other" staff.  This is the formatter's version of
								// the "xbstem" flag in the NOTE node; xotherstaff is
								// sometimes set when xbstem is not (for example, if the
								// note is stem-down and is being moved because a higher
								// pitch note already had xbstem set) and it is sometimes
								// not set when xbstem is (for example, if the note is
								// stem-down and it is on the last staff in the system,
								// xbstem is ignored).  See comment in placecd.c for more
								// information.
		UNBFLD	Xunusualstem:1;	// the note is on the unusual side of the stem (because
								// it is part of a clustered chord)

		INT16	Xaccx;			// Accidental position (tenth point units):
								//	 Accidental x location is (xhdx - xaccx);
								// Temporary values in placecd():
								//	 Accidental stacking column is stored in placecd();
								//	 then changed to accidental x offset described above.

		INT16	Xstemxoff;		// x location of stem, offset from xhdx;
								// the xstemx(ptr) macro is xhdx(ptr)+xstemxoff(ptr).
		XYPOS	Xscrhdx;		// value of "xhdx" in offscreen image
		XYPOS	Xscrhdy;		// value of "xhdy" in offscreen image; these values
								// are set from xhdx and xhdy when formatting is
								// complete.  During formatting, xscrhdx and
								// xscrhdy retain the previous values of xhdx and
								// xhdy: scrhdx and scrhdy indicate the note
								// location on the (outdated) screen display.
								// This allows insertion point selection
								// during formatting.  Values of NOT_ON_SCREEN
								// indicate the note is not on the offscreen image.
	
		INT8	Xnote_pruned;	// Note is not in the staff list when the music is formatted;
								// this is set when a note (usually a rest) is not plotted at all.
		INT8	Xnote_hasannos;	// Set to flag that a Note has annotations that may need plotting.
	
	// 36 bytes to here
	
	// These fields are actually pointers.  They have, of course, no relevance whatsoever
	// if this node is saved to disk (as part of graphics save).
	//
		DefTiliaPtr Xpback;		// staff list back pointer
		DefTiliaPtr Xpnext;		// staff list next pointer. The staff list contains only notes and rests
	
} NTDATA_node;
#define NTDATA_sizes "11(1111121)(341)11111111xx1(221111)22xx11"

enum{
	NOT_ON_SCREEN	= UNDEFXY
};



//								=== CDDATA ===
// The CDDATA node (Chord Data Node) is a temporary node for printing.
//
// The CDDATA node immediately follows after the NTDATA node of the last note
// in a chord linked by staff-list pointers (xpnext and xpback). It is not
// directly pointed to by xpnext or xpback pointers; it is in the voice-list
// only.
//
// The xuphiy, xuploy, xdownhiy, xdownloy fields are accessed via the routines
// set_hiy, set_loy, get_hiy, get_max_hiy, get_loy, get_min_loy.
//
enum{						// Xtup_mark values
	TUP_MARK_BEFORE	= 2,	// tuplet mark should be printed between this note and the previous note	
	TUP_MARK_HERE	= 1,	// tuplet mark should be printed at this note
	NO_TUP_MARK		= 0		// no tuplet mark drawn at or before this chord
};

typedef struct {
		LINKS				// macro to make next and back pointers for Lime
		UNINT8	Xkind;		// kind of node value is CDDATA
		INT8	Xcontext;	// notation context, should always be 0, all contexts.

		XYPOS	Xuphiy;		// highest y location used in the stem-up	chord
		XYPOS	Xuploy;		// lowest  y location used in the stem-up	chord
		XYPOS	Xdownhiy;	// highest y location used in the stem-down chord
		XYPOS	Xdownloy;	// lowest  y location used in the stem-down chord
		XYPOS	Xorigin;	// relocation origin to use for the chord; based
							// on widths of objects and ignoring note lengths
							// (see also Xidealx below, and comments in
							// relocate.c)

		XYPOS	Xprehiy;	// highest y for objects preceding the chord; after
							// formatting completes, changed to highest y in system
		XYPOS	Xpreloy;	// lowest y for objects preceding the chord; after
							// formatting completes, changed to lowest y in system
		XYPOS	Xprex;		// x loc used by preceding xprehiy/xpreloy objects;
							// this field is ignored unless it is non-zero

	// 34 bytes to here, excluding links
		UNBFLD	Xoppos : 1;		// TRUE if there are notes in both stem directions
								// which are sustained during this chord; this means
								// slurs, staccatos, fermatas etc. should
								// be plotted at the end of the stem instead of
								// by the notehead.
		UNBFLD	Xtup_start : 1; // start of a tuplet group (see xtup_info field in NTDATA)
		UNBFLD	Xtup_end : 1;	// end of a tuplet group (see xtup_info field in NTDATA)
		UNBFLD	Xtup_mark : 2;	// tuplet mark plotting info  (see xtup_info in NTDATA);
								// NO_TUP_MARK, TUP_MARK_HERE, or TUP_MARK_BEFORE;

		UNBFLD	Xupbstem : 1;	// TRUE if upstem goes between staves (up to next higher
								// staff); see the xbstem field in the NOTE node and
								// comments in placecd.c
		UNBFLD	Xdnbstem : 1;	// TRUE if down stem goes between staves (to next lower
								// staff); see the xbstem field in the NOTE node and
								// comments in placecd.c
		UNBFLD	Xnewmeas : 1;	// TRUE if this chord is at the start of a new measure

		UNINT8	Xcdclef;		// type of clef for chord; see CLEF node for definitions

		UNINT16	Xrcmeas;		// for multi-measure rests: number of measures that
								// are combined.  This field is set only if the
								// notes in the chord have xrestchar==RCMEAS.

		UNBFLD	Xupleft	 : 3;	// beam parts to left for stem up (used by MusicXML write)
		UNBFLD	Xupright : 3;	// beam parts to right for stem up (used by MusicXML write)
		UNBFLD	Xslurstart : 2; // slur start (used by MusicXML write):
								// UP = start of slur above notes (or for upper voice)
								// DOWN = start of slur below notes (or for lower voice)
								// UPORDOWN = start of two slurs (like UP and DOWN combined)

		UNBFLD	Xdnleft	 : 3;	// beam parts to left for stem down
		UNBFLD	Xdnright : 3;	// beam parts to right for stem down
		UNBFLD	Xslurend : 2;	// slur end (used by MusicXML write):
								// UP = end of slur above notes (or for upper voice)
								// DOWN = end of slur below notes (or for lower voice)
								// UPORDOWN = end of two slurs (like UP and DOWN combined)
		INT16	Xlorchorx;		// lowest x location RELATIVE to chord ORIGIN, including accidentals,
								// excluding width of note heads to the left.
		UNINT8	Xlorheadx;		// width of note head(s) to the left of chord origin
		INT8	Xlormorex;		// additional (as yet unused) field in case we need it
		INT8	Xpstaff;		// pstaff number for this chord
		INT8	Xopposhide;		// TRUE if opposite stem directions should be ignored for marks,
								// typically because there are two notes one of which is a
								// hidden rest and there is no need to replace marks, etc.
	// 46 bytes to here excluding links
} CDDATA_node;
#define CDDATA_sizes "11xxxxxxxx(1112111)12(332)(332)21111"


//								=== CDDATA2 ===
// The CDDATA2 node is an extension of the CDDATA node
//
// Every CDDATA node has a CDDATA2 and CDDATA3 node immediately following it.  The
// CDDATA2 and CDDATA3 nodes contain information fields which did not fit into the
// CDDATA node.  The pointer always points at the CDDATA node; the CDDATA2 and CDDATA3
// fields are accessed through the CDDATA node's xnext pointer.
//
typedef struct {
		LINKS					// macro to make next and back pointers for Lime
		UNINT8	Xkind;			// kind of node value is CDDATA
		INT8	Xcontext;		// notation context; 0 all contexts, -1 not score

		INT32	Xswholec;		// Printed whole note count since start of
								// system; format is: 16'.16'
		INT32	Xmwholec;		// Printed whole note count since start of
								// measure; format is: 16'.16'
		INT8	Xabovebeam[2];	// Count of subbeam lines (secondary beams) drawn above the main beam line,
								// for UP and DOWN stems.  This helps place subbeam lines
								// for LH/RH grand staff switches.
		INT8	Xbelowbeam[2];	// Count of subbeam lines drawn below the main beam line,
								// for UP and DOWN stems.  This helps place subbeam lines
								// for LH/RH grand staff switches.
		INT8	unused1;		// used to be Xpstaff number (moved to CDDATA for efficiency)
		INT8	Xplayer;		// layer number for this chord
		XYPOS	Xidealx;		// Ideal origin for chord, based on note lengths and
								// ignoring widths of objects (see also Xorigin above).
		INT8	unused2;		// used to be Xopposhide (moved to CDDATA for efficiency)
		INT8	Xreloc_zone;	// In relocation pass: Relocation zone for this chord;
								// if there are no ZONE nodes, this is always zero.
		INT16	Xreloc_spring;	// In spacing pass: set to NOSPRING for notes without springs
								// In relocation pass: Number of the spring starting here, or NOSPRING
		INT16	XchordNum;		// by system; for finding corresponding chord in other staff/layer;
								// this is set in relink.c
		// 26 bytes to here
		INT8	unused[10];
		
		// Pointers only ever used internally. JA20150110 7.5.311. Moved from CDDATA3.
		//
		DefTiliaPtr Xtsnext;	// next cddata node in timeslice; or first cddata node
		//						// in next timeslice if this is last in this timeslice
		DefTiliaPtr Xtsback;	// previous cddata node in timeslice; or last cddata node
		//						// in previous timeslice if this is first in this timeslice
		
		DefTiliaPtr Xbar2left;	// Previous known BAR node
		DefTiliaPtr Xbar2rght;	// Next known BAR node	[may-not be needed]
	
} CDDATA2_node;
#define CDDATA2_sizes "1144111111x1122"	// excludes the pointers at end

enum{
	NOSPRING = UNDEF16
};


//								=== CDDATA3 ===
// The CDDATA3 node was an extension of the CDDATA and CDDATA2 nodes
// It is now obsolete because, with the increased node size, everything fits
// comfortably into just 2 nodes (CDDATA and CDDATA2).	// JA20150110 7.5.311. 
//
// CDDATA3 is reserved, however, in case it is needed in the future.  If it is used,
// it is recommended that it should not contain anything that could possibly be
// relevant on disk; indeed it should not be written to disk even in graphics save.
//
typedef struct {
	LINKS					// macro to make next and back pointers for Lime
	UNINT8	Xkind;			// kind of node value is CDDATA
	INT8	Xcontext;		// notation context; 0 all contexts, -1 not score
	
} CDDATA3_node;
#define CDDATA3_sizes "11"
#define CDDATA3_necessary 0


//						=== comments for TPARAM nodes===
// During formatting, a PDATA node gets inserted between the TPARAM node and
// the following TTEXT node.  The PDATA node acts as an extension of the TPARAM
// node (as well as the LINE node).  PDATA nodes are inserted only after TPARAM
// nodes which do not have xadjacent set -- PDATA nodes contain information for
// the whole text string, including font changes.
//



//								=== PDATA ===
// The PDATA nodes store position data for annotations.  They are used only during
// print formatting (ie, they are not stored on disk).  A PDATA node follows
// TPARAM and LINE nodes on the current page of music.  It stores the location of
// the text/line, so that the user can select it for editing etc.
//
// Every LINE node active in the current Notation Context will be followed by a PDATA
// node during formatting. TPARAM nodes are also followed by PDATA nodes, unless the
// TPARAM node has xadjacent set; see the definition of TPARAM nodes above.
//
// Access to the fields of the PDATA node are via the preceding TPARAM or LINE node.
// This is similar to how NTDATA fields are accessed from the NOTE node (the
// macro definitions below contain an "xnext" call).
//
typedef struct {
		LINKS					// macro to make next and back pointers for Lime
		UNINT8	Xkind;			// kind of node value is CDDATA
		INT8	Xcontext;		// notation context; 0 all contexts, -1 not score

		XYPOS	Xx1;			// x location of start of item (Tilia coordinates)
		XYPOS	Xy1;			// y location of start of item (Tilia coordinates)
		XYPOS	Xx2;			// x location of end of item (Tilia coordinates)
		XYPOS	Xy2;			// y location of end of item (Tilia coordinates)
		INT8	Xy1pstaff;		// pstaff (y origin) for xy1
		INT8	Xy2pstaff;		// pstaff (y origin) for xy2
		XYPOS	Xscrx1;			// value of "xx1" currently displayed (Tilia coordinates)
		XYPOS	Xscry1;			// value of "xy1" currently displayed (Tilia coordinates)
		XYPOS	Xscrx2;			// value of "xx2" currently displayed (Tilia coordinates)
		XYPOS	Xscry2;			// value of "xy2" currently displayed (Tilia coordinates).
								// These xscr... values are analogous to the
								// use of xhdx/xscrhdx for NOTE nodes:
								// when formatting is complete, xx1 is copied to
								// xscrx1 etc.
								//	During formatting, xscrx1, xscry1 etc retain the
								// previous values of xx1, xy1: they indicate the
								// item location on the (outdated) screen display.
								// This allows item selection when print-formatting
								// is interrupted.
								// Values of NOT_ON_SCREEN indicate the item is not
								// on the offscreen image.
		INT16	Xchyphen;		// non-zero indicates centered hyphen must be drawn after this
								// annotation; the value is the width of the hyphen character.
								// This is set by the drawanno pass of formatting,
								// and is only used by the drawhyp pass of formatting.
		INT8	Xanno_selected;	// used by the editor to mark selected annotations for
								// GROUP_SELECT when MODE!=MUSIC_MODE
		INT8	unused;
	
		UNINT16	Xpdataclipl;	// For lines and curves, if non-zero, temporary clip offset forwards from start
		UNINT16	Xpdataclipr;	// For lines and curves, if non-zero, temporary clip offset backwards from end 
		// 44 bytes to here
} PDATA_node;
#define PDATA_sizes "11xxxx11xxxx21122"



//							   === XMLT ===
// XMLT nodes are created by the Import MusicXML menu option.
// They store timing information from the MusicXML representation; they are only
// used during the import process.  Also used during NIFF import
//

typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is XMLT
	INT8	Xcontext;			// defines the notation contexts this node applies to

	INT32	Xnwholec;			// MusicXML whole note count for note's timeslice
	INT32	Xxmltime;			// MusicXML import internal use only: time in MusicXML units
	INT32	Xxmlincr;			// MusicXML import internal use only: chord duration in MusicXML units
	INT32	Xxmldivisions;		// MusicXML import internal use only: MusicXML units per quarter note
	INT8	Xxmlstopper;		// MusicXML import internal use only: TRUE to not backup over this (measure boundary)
	INT8	Xnifft;				// not FALSE if this was created during NIFF import
} XMLT_node;
#define XMLT_sizes  "11444411"
#define NIFFT_sizes XMLT_sizes






// ------------------------ Graphics Element list declarations ------------------------
//
// These are used internally for plotting graphical elements.  They are written to disk
// when saving graphical nformation.

// Four Tilia node kinds are used in Graphics Element list:
//		GE_CHAR, GE_STRING, GE_LINE, GE_CURVE.
//
// These node kinds are different from other Tilia nodes, because they have a
// flag byte where "normal" Tilia nodes have the xcontext field.
//
// Individual characters are represented in an GE_CHAR record.
//
// Strings of arbitrary length are represented by the following sequence:
//		- one or more GE_STRING records which give the text of the string
//		- an GE_CHAR record which defines the font, size, style
//		//and location of the string.
//
// Lines are represented in an GE_LINE record.
//
// Parabolas are represented in an GE_CURVE record.
//
//
// The prefix "G" for "Graphics Element" is used on all field names.
//
// Locations are given as (x, y, gpstaff).
// The "gpstaff" field records the staff number that the Graphics Element is
// associated with.
//

//								--- GE_FLAGS_BYTE ---
// Macro to define the flag byte common to all GE nodes.  This flag byte is
// in the same location as the context field is for other Tilia nodes(!).
//
// GEs are usually drawn in "or" mode.
// However, it is possible to have GEs that are drawn in "bit-clear" mode.  These
// are tagged by Gmode = BICMODE.
//
#define GE_FLAGS_BYTE																	  \
	UNBFLD Gmode	: 2;		/* drawing mode: ORMODE, BICMODE						*/\
	UNBFLD Ganno	: 2;		/* nonzero if Graphics Element is from an annotation:	*/\
								/* FROM_ANNO if from non-hidden annotation;				*/\
								/* HIDDEN_SCREEN or HIDDEN_PRINT if hidden anno			*/\
	UNBFLD Gremove	: 1;		/* TRUE if GE is removed due to Allow Any Option-Click	*/\
	UNBFLD Gscreen	: 1;		/* TRUE if GE for screen only (next GE for print)		*/\
	UNBFLD unused	: 2;

enum{
	FROM_ANNO		= 1,		// ganno value: GE is from an annotation
	HIDDEN_PRINT	= 2,		// ganno value: GE is from anno hidden for print
	HIDDEN_SCREEN	= 3 		// ganno value: GE is from anno hidden on screen
};




//								=== NOTEPTR === (used in GE list)
//
// The NOTEPTR node is used in the GE list, to point at the Tilia NOTE which is
// associated with the GE nodes that follow.  NOTEPTR nodes need no xcontext field
// nor do they need the GE_FLAGS_BYTE.  This matters since the NOTEPTR node appears
// in the GE lists, but the type checking macros do not consider it a GE node.
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is NOTEPTR
	INT8	Xcontext;			// Always zero for GE nodes, etc.
	
	DefTiliaPtr Xnoteptr;		// NOTE associated with following GE nodes
	
	INT8	unused[44];			// unused bytes in this node
	
	INT8	gunused;
	INT8	gunused_flags;		// Always zero.
} NOTEPTR_node;
#define NOTEPTR_sizes "114"



//								=== GE_CHAR ===
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8		Xkind;			// kind of node; value is GE_CHAR
	INT8		Xcontext;		// Always zero for GE nodes, etc.

	XYPOS		Gcharx, Gchary;	// x,y location in Tilia coordinates;
								// x,y == ADJACENT means continue at current location.
								// ADJACENT only occurs if ganno is set, since it
								// is only generated by user annotations with font changes
	INT8		Gcharps;		// pstaff number, used for y relocation (staff moves)
	UNINT8		Gcharvar;		// variety of GE_CHAR node (see tilia.h)
	INT16		Gcfont;			// font number; this is Mac font number, without
								// Tilia's font number remapping.
	UNINT8		Gcharset;		// Windows charset value for foreign language support
	UNINT8		Gcsize;			// font size
	INT16		Gcharnum;		// character number; if the preceding record is an GE_STRING,
								// then the gcharnum field is ignored.
	TiliaStyle	Gcstyle;		// italic, bold etc 

	UNBFLD		Gpslayout   :1;	// TRUE if PostScript LineLayoutOn should be used on the text
	UNBFLD		ungused4	:4;
	UNBFLD		Gchar_yreloc:2;	// YWITHSTAFF or YFIXED
	UNBFLD		Gcharhide	:1;	// from a special hidden annotation

	INT8		Gsuper;			// amount to superscript (positive) or subscript (negative)
	INT8		Gexpand;		// amount to expand (move right) or condense (move left)
	
	INT8		Gunicode;		// if the preceding record is an GE_STRING, flags it is Unicode (UTF-8)
								// (<=-16 or >=+16 reserved for wide character UTF-16; but unlikely)
	
	INT8		Gcharcolor;		// color index of the character ((palette << 4) | index)
	
	// 24 bytes to here
	INT8		unused1[2];
	INT16		unused2[2];
	INT32		unused4[4];
		
	TiliaRGBA	Gchar_rgba;		// Reserved for generalized colouring
	//
	// 50 bytes to here
	
	INT8		gunused;
	GE_FLAGS_BYTE				// defines flags common to all GE nodes; 52nd byte.
} GE_CHAR_node;
#define GE_CHAR_sizes "11xx1121121(1421)11111122444411111(22112)"


//								=== GE_STRING ===
//
enum{
	GE_STRLEN	= DATA_BYTES_IN_A_TILIA_NODE-2	// max number of character bytes in a GE_STRING node
};

typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is GE_STRING
	INT8	Xcontext;			// Always zero for GE nodes, etc.

	char	Gtext[GE_STRLEN];	// text of the string
	char	Gtext0;				// Always zero (and should be forced to zero if node is read from disk)

	GE_FLAGS_BYTE				// defines flags common to all GE nodes; 52nd byte.	
} GE_STRING_node;
#define GE_STRING_sizes "111111111111111111111111111111111111111111111111111(22112)"
//						^         ^         ^         ^         ^         ^


//								=== GE_LINE ===
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is GE_LINE
	INT8	Xcontext;			// Always zero for GE nodes, etc.

	XYPOS Glinex1, Gliney1;		// x,y start location in Tilia coordinates
	INT8  Glineps1;				// start pstaff number, used for y relocation (staff moves)

	INT8  Glineps2;				// end pstaff number, used for y relocation (staff moves)
	XYPOS Glinex2, Gliney2;		// x,y end location in Tilia coordinates

	XYPOS Gclipx1, Gclipy1;		// location of first clip point
	XYPOS Gclipx2, Gclipy2;		// location of second clip point
	// 36 bytes to here
	
	INT8   Glinecolor;			// color index of the line ((palette << 4) | index)
	UNBFLD gline_unused4 : 4;
	UNBFLD Gyreloc : 2;			// y relocation for the first point: values
								// are YWITHSTAFF, YHALFABOVE, YHALFBELOW, YFIXED.
								// These constants are declared in format.h.
								// This field is used for staff-moving.
								// In a grand staff, the grand-staff beams and the
								// stems that go to them have to move half as
								// much as the rest of the staff moves.
								// See GE_yrelocate()
	UNBFLD Gyreloc2 : 2;		// y relocation for the second point

	UNBFLD Glinehide : 1;		// TRUE if line should be drawn in hidden style
	UNBFLD Glinevar  : 7;		// variety of GE_LINE node

	UNBFLD Glinedash : 4;		// dashed lines: length of dashes in dots;
								// zero (NODASH) indicates a solid line
	UNBFLD Glinegap	 : 4;		// dashed lines: distance between dashes;
								// zero (NODASH) indicates a solid line
	
	INT16  Glinewidth;			// width of line in tenth point units;
	
	// 42 bytes to here
	INT8		unused1[2];
	INT16		unused2;
	
	TiliaRGBA	Gline_rgba;		// Reserved for generalized colouring
	//
	// 50 bytes to here
	
	INT8		gunused;
	GE_FLAGS_BYTE				// defines flags common to all GE nodes; 52nd byte.	
} GE_LINE_node;
#define GE_LINE_sizes "11xx11xxxxxx1(422)(17)(44)211211111(22112)"

enum{
	DEFAULT_CLIP_W_OFFSET	= 50,	// Default clip offset, over and above line width
	DEFAULT_CLIP_OFFSET		= 280	// used to make default clip rectangle for line; large enough to clear fattest line
};


//								=== GE_CURVE ===
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is GE_CURVE
	INT8	Xcontext;			// Always zero for GE nodes, etc.

	XYPOS Gcurvex1, Gcurvey1;	// x,y start location in Tilia coordinates
	INT8  Gcurveps1;			// start pstaff number, used for y relocation (staff moves)

	INT8  Gcurveps2;			// end pstaff number, used for y relocation (staff moves)
	XYPOS Gcurvex2, Gcurvey2;	// x,y end location in Tilia coordinates

	XYPOS Gc1x, Gc1y;			// location of first control point
	
	INT8  Gcurvecolor;			// color index of the curve ((palette << 4) | index)

	UNBFLD Gcurvedash : 4;		// dashed curves: length of dashes in dots;
								// zero (NODASH) indicates a solid curve
	UNBFLD Gcurvegap  : 4;		// dashed curves: distance between dashes;
								// zero (NODASH) indicates a solid curve

	XYPOS Gc2x, Gc2y;			// location of second control point

	UNBFLD Gfat : 1;			// TRUE if the curve becomes fatter in the middle
	UNBFLD unused4bits : 4;
	UNBFLD Gcurvehide  : 1;		// TRUE if line should be drawn in hidden style
	UNBFLD unused2bits : 2;		

	UNINT8 Gcurvevar;			// variety of GE_CURVE node 
	
	// 40 bytes to here
	INT8		unused1[2];
	
	UNINT16		Gcurveclipl;	// Distance from start of left  clip point (0 means no clipping)
	//							// Used at the start or middle when a resizable line extends over a system break
	UNINT16		Gcurveclipr;	// Distance from  end  of right clip point (0 means no clipping)
	//							// Used at the  end  or middle when a resizable line extends over a system break
	
	TiliaRGBA	Gcurve_rgba;	// Reserved for generalized colouring	
	// 
	// 50 bytes to here
	
	INT8		gunused;
	GE_FLAGS_BYTE				// defines flags common to all GE nodes; 52nd byte.	
} GE_CURVE_node;
#define GE_CURVE_sizes "11xx11xxxx1(44)xx(1412)1112211111(22112)"



//								=== GE_GENERIC ===
// The GE_GENERIC node is defined as a programming convenience; no nodes of kind
// GE_GENERIC actually occur in Tilia files.  It is used to access a GE node of
// unknown kind, for the flag fields common among the GE nodes.
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is GE_CURVE
	INT8	Xcontext;			// Always zero for GE nodes, etc.

	INT8	Xbyte[DATA_BYTES_IN_A_TILIA_NODE-2];
	//
	// 50 bytes to here
	
	INT8	gunused;
	GE_FLAGS_BYTE				// defines flags common to all GE nodes; 52nd byte.	
} GE_GENERIC_node;




//								=== GENERIC ===
// The GENERIC node is defined as a programming convenience; no nodes of kind
// GENERIC actually occur in Tilia files.  To access a node of unknown kind, the
// code such as the following can be used to test the kind of the node and do a
// C typecast.
//		GENERIC_node *g;
//		NOTE_node *n;
//		...
//		if (generic_node->Xkind==NOTE) {
//			n = (note_node)g;
//			... }
//
typedef struct {
	LINKS						// macro optionally makes next/back pointers for Tilia nodes
	UNINT8	Xkind;				// kind of node; value is anything
	INT8	Xcontext;			// defines the notation contexts this node applies to

	INT8	Xbyte[DATA_BYTES_IN_A_TILIA_NODE];
} GENERIC_node;


// restore packing
#if defined( __BORLANDC__ )
	#pragma option -a.
#else
	#pragma pack(pop)
#endif

#endif
